/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2017
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.user.persistence;

import java.sql.SQLException;
import java.util.Collections;
import java.util.List;
import java.util.function.Supplier;

import javax.sql.DataSource;

import com.google.common.base.Suppliers;

import jakarta.validation.constraints.NotNull;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.Item;
import net.bluemind.core.container.persistence.BooleanCreator;
import net.bluemind.core.container.persistence.StringCreator;
import net.bluemind.core.container.repository.IItemStore;
import net.bluemind.core.jdbc.JdbcAbstractStore;
import net.bluemind.core.rest.BmContext;
import net.bluemind.repository.provider.RepositoryProvider;
import net.bluemind.user.repository.IUserSubscriptionStore;

public class UserSubscriptionStore extends JdbcAbstractStore implements IUserSubscriptionStore {
	private final Supplier<IItemStore> itemStore;

	public UserSubscriptionStore(BmContext context, DataSource dataSource, Container container) {
		super(dataSource);
		itemStore = Suppliers.memoize(() -> RepositoryProvider.instance(IItemStore.class, context, container));
	}

	public void subscribe(String subject, Container container) throws SQLException {

		Item item = itemStore.get().get(subject);
		if (item == null) {
			return;
		}

		insert("INSERT INTO t_container_sub (container_uid, container_type, user_id) VALUES (?, ?, ?)", container,
				(con, statement, index, currentRow, value) -> {
					statement.setString(index++, value.uid);
					statement.setString(index++, value.type);
					statement.setLong(index++, item.id);
					return index;
				});
	}

	public boolean isSubscribed(String subject, Container container) throws SQLException {
		Item item = itemStore.get().get(subject);
		if (item == null) {
			return false;
		}
		String query = "SELECT 1 FROM t_container_sub WHERE container_uid = ? AND user_id = ?";
		return unique(query, BooleanCreator.FIRST, Collections.emptyList(),
				new Object[] { container.uid, item.id }) != null;
	}

	public void unsubscribe(String subject, String containerUid) throws SQLException {
		Item item = itemStore.get().get(subject);
		if (item == null) {
			return;
		}
		delete("DELETE FROM t_container_sub WHERE container_uid = ? AND user_id = ?",
				new Object[] { containerUid, item.id });
	}

	public void unsubscribeAll(String subject) throws SQLException {
		Item item = itemStore.get().get(subject);
		if (item == null) {
			return;
		}

		delete("DELETE FROM t_container_sub WHERE user_id = ?", new Object[] { item.id });
	}

	/**
	 * @param subject user uid
	 * @param type    might be null to search all subscriptions
	 * @return a list of container uid the subject is subscribed to
	 * @throws SQLException
	 */
	public List<String> listSubscriptions(@NotNull String subject, String type) throws SQLException {

		Item item = itemStore.get().get(subject);
		if (item == null) {
			return Collections.emptyList();
		}

		String query = "SELECT container_uid FROM t_container_sub WHERE user_id = ?";
		Object[] params = new Object[] { item.id };
		if (type != null) {
			query += " AND container_type = ?";
			params = new Object[] { item.id, type };
		}

		return select(query, StringCreator.FIRST, Collections.emptyList(), params);
	}

	public List<String> subscribers(@NotNull String containerUid) throws SQLException {
		String query = "SELECT ci.uid FROM t_container_sub "//
				+ "INNER JOIN t_container_item ci ON ci.id=user_id "//
				+ "WHERE container_uid = ?";
		return select(query, StringCreator.FIRST, Collections.emptyList(), new Object[] { containerUid });
	}

	public void allowSynchronization(String subject, Container container, boolean sync) throws SQLException {

		Item item = itemStore.get().get(subject);
		if (item == null) {
			return;
		}

		String updateQuery = "UPDATE t_container_sub SET offline_sync = ? WHERE container_uid = ? AND user_id = ?";
		update(updateQuery, null, (con, statement, index, currentRow, value) -> {
			statement.setBoolean(index++, sync);
			statement.setString(index++, container.uid);
			statement.setLong(index++, item.id);
			return index;
		});
	}

	public boolean isSyncAllowed(String subject, Container container) throws SQLException {
		Item item = itemStore.get().get(subject);
		if (item == null) {
			return false;
		}

		String query = "SELECT offline_sync FROM t_container_sub WHERE container_uid = ? AND user_id = ?";
		Boolean ret = unique(query, BooleanCreator.FIRST, Collections.emptyList(),
				new Object[] { container.uid, item.id });
		return Boolean.TRUE.equals(ret);
	}

	public void updateAutomount(String subject, Container container, boolean automount) throws SQLException {
		Item item = itemStore.get().get(subject);
		if (item == null) {
			return;
		}

		String updateQuery = "UPDATE t_container_sub SET automount = ? WHERE container_uid = ? AND user_id = ?";
		update(updateQuery, null, (con, statement, index, currentRow, value) -> {
			statement.setBoolean(index++, automount);
			statement.setString(index++, container.uid);
			statement.setLong(index++, item.id);
			return index;
		});
	}

	public boolean isAutomounted(String subject, Container container) throws SQLException {
		Item item = itemStore.get().get(subject);
		if (item == null) {
			return false;
		}

		String query = "SELECT automount FROM t_container_sub WHERE container_uid = ? AND user_id = ?";
		Boolean ret = unique(query, BooleanCreator.FIRST, Collections.emptyList(),
				new Object[] { container.uid, item.id });
		return Boolean.TRUE.equals(ret);
	}
}
