import parse from "emailjs-mime-parser";
import { asn1, pkcs7, util } from "node-forge";

import { MessageBody } from "@bluemind/backend.mail.api";
import { envelope, parser, helper } from "@bluemind/smime.commons";

import extractSignedData from "../smime/helpers/SMimeSignedDataParser";

const { getSignedDataEnvelope } = envelope;
const { initializePkcs7Parser } = parser;
const { hasAttachedSignature } = helper;

export default function parseSignedEml(
    bodyStructure: MessageBody.Part,
    eml: string
): { content: string; toDigest: string; envelope: pkcs7.Captured<pkcs7.PkcsSignedData> } {
    const isOpaque = hasAttachedSignature(bodyStructure, eml);
    const { envelope, toDigest } = isOpaque
        ? getEnvelopeFromAttachedSignature(eml)
        : getEnvelopeFromDetachedSignature(eml);
    return { envelope, toDigest, content: isOpaque ? toDigest : eml };
}

function getEnvelopeFromAttachedSignature(eml: string): {
    envelope: pkcs7.Captured<pkcs7.PkcsSignedData>;
    toDigest: string;
} {
    initializePkcs7Parser();

    const rootNode = parse(eml);
    const buffer = new util.ByteStringBuffer(rootNode.content);
    const asn1Data = asn1.fromDer(buffer);

    const toDigest = getContentFromAttachedSignature(eml);

    return { envelope: <pkcs7.Captured<pkcs7.PkcsSignedData>>pkcs7.messageFromAsn1(asn1Data), toDigest };
}

function getEnvelopeFromDetachedSignature(eml: string) {
    const { toDigest, pkcs7Part } = extractSignedData(eml);
    const envelope = getSignedDataEnvelope(pkcs7Part);
    return { toDigest, envelope };
}

export function getContentFromAttachedSignature(eml: string): string {
    const parsed = parse(eml);
    const asn1Data = asn1.fromDer(new util.ByteStringBuffer(parsed.content));
    const msg = pkcs7.messageFromAsn1(asn1Data);

    function getValue(obj: asn1.Asn1): string {
        return obj.composed ? (obj.value as asn1.Asn1[]).map(getValue).join("") : (obj.value as string);
    }

    return getValue(msg.rawCapture.content);
}
