import { MailboxItemsClient, MailboxItem, MessageBody } from "@bluemind/backend.mail.api";
import { MimeType } from "@bluemind/email"; // FIXME: move MimeType into @bluemind/mime
import { extensions } from "@bluemind/extensions";
import { MimeBuilder } from "@bluemind/mime";
import session from "@bluemind/session";
import { constants, enums, exceptions } from "@bluemind/smime.commons";

import pkcs7 from "../pkcs7";
import { getMySignatureKeyPair } from "../pki";

import getRemoteContentFn from "./helpers/getRemoteContentFn";
import getSMimeImplementation from "./helpers/getSMimeImplementation";
import buildSignedEml from "./helpers/SMimeSignedEmlBuilder";

const { SMIME_SIGNATURE_ERROR_PREFIX } = constants;
const { SmimeErrors } = exceptions;
const { CRYPTO_HEADERS } = enums;

export default async function (item: MailboxItem, folderUid: string): Promise<MailboxItem> {
    try {
        item.body.structure = removePreviousSignedPart(item.body.structure!);

        const client = new MailboxItemsClient(session.sid, folderUid);
        const unsignedMimeEntity = await new MimeBuilder(getRemoteContentFn(item.imapUid!, folderUid)).build(
            item.body.structure!
        );

        const signedContent = await doSign(unsignedMimeEntity);
        const eml = buildSignedEml(unsignedMimeEntity, signedContent, item.body);
        const address = await client.uploadPart(eml);
        item.body.structure = { address, mime: "message/rfc822", children: [] };
    } catch (error) {
        const errorCode = error instanceof SmimeErrors ? error.code : CRYPTO_HEADERS.UNKNOWN;
        throw `[${SMIME_SIGNATURE_ERROR_PREFIX}:${errorCode}]` + error;
    }
    return item;
}

async function doSign(entity: string): Promise<string> {
    const implementation = getSMimeImplementation("signMimeEntity");
    return implementation(entity);
}

extensions.register("smime.implementations", "smime.service-worker", {
    signMimeEntity: {
        fn: async function (entity: string) {
            const { key, certificate } = await getMySignatureKeyPair();
            return await pkcs7.sign(entity, key, certificate);
        },
        priority: 0
    }
});

function removePreviousSignedPart(structure: MessageBody.Part): MessageBody.Part {
    if (structure.mime === MimeType.MULTIPART_MIXED && structure.children) {
        const signedPartIndex = structure.children.findIndex(part => part.mime === MimeType.PKCS_7_SIGNED_DATA);
        if (signedPartIndex !== -1) {
            structure.children.splice(signedPartIndex, 1);
            if (structure.children.length === 1) {
                return structure.children[0];
            }
        }
    }
    return structure;
}
