import { registerRoute } from "workbox-routing";
import { pki } from "node-forge";
import { extensions } from "@bluemind/extensions";
import BmRoles from "@bluemind/roles";
import { clear as clearPki, getMyStatus, setMyPrivateKeys, setMyCertificates } from "./pki";
import bodyDB from "./smime/cache/SMimeBodyDB";
import SMimeApiProxy from "./SMimeApiProxy";
import { constants, enums } from "@bluemind/smime.commons";
import { checkRecipientEmail, isPemReadable as pemToCertificate } from "./pki/checkCertificate";
import { isKeyReadable } from "./pki/checkKey";
import { MimeType } from "@bluemind/email";

const { SMIME_INTERNAL_API_URL, SMIME_UNTRUSTED_CERTIFICATE_ERROR_PREFIX } = constants;
const { PKIEntry } = enums;

extensions.register("serviceworker.handlers", "smime-plugin", {
    "api-handler": { class: SMimeApiProxy, priority: 256, role: BmRoles.CAN_USE_SMIME }
});

registerRoute(SMIME_INTERNAL_API_URL, hasCryptoFilesHandler, "GET");
registerRoute(SMIME_INTERNAL_API_URL, clearPkiAndCache, "DELETE");
registerRoute(`${SMIME_INTERNAL_API_URL}/${PKIEntry.PRIVATE_KEYS}`, setPrivateKeys, "PUT");
const setCertificatesMatcher = ({ url }) => {
    return url.pathname === `${SMIME_INTERNAL_API_URL}/${PKIEntry.CERTIFICATES}`;
};
registerRoute(setCertificatesMatcher, setCertificates, "PUT");

async function hasCryptoFilesHandler() {
    return new Response(await getMyStatus());
}

async function clearPkiAndCache() {
    await clearPki();
    await bodyDB.clearBody();
    await bodyDB.clearGuid();
    await caches.delete("smime-part-cache");

    return new Response();
}

async function setPrivateKeys({ request }) {
    const blob = await request.blob();
    const pemKeys = await blob.text();
    try {
        const keys = JSON.parse(pemKeys).filter(isKeyReadable);
        await setMyPrivateKeys(new Blob([JSON.stringify(keys)], { type: MimeType.JSON }));
        return new Response();
    } catch (error) {
        return new Response(error, { status: 500 });
    }
}

async function setCertificates({ request }) {
    const expectedAddress = new URL(request.url).searchParams.get("email");
    const blob = await request.blob();
    const certificatesText = await blob.text();
    let certificates;
    try {
        certificates = JSON.parse(certificatesText).map(pemToCertificate);
    } catch (err) {
        return new Response(err, { status: 500 });
    }

    let error;
    const checkedCertificates = certificates
        .filter(certificate => {
            try {
                checkRecipientEmail(certificate, expectedAddress);
                return true;
            } catch (err) {
                error = err;
                return false;
            }
        })
        .map(pki.certificateToPem);
    if (checkedCertificates.length === 0 && error) {
        const errorMessage = `[${SMIME_UNTRUSTED_CERTIFICATE_ERROR_PREFIX}:${error.code}]` + error.message;
        return new Response(errorMessage, { status: 500 });
    }

    await setMyCertificates(new Blob([JSON.stringify(checkedCertificates)], { type: MimeType.JSON }));
    return new Response();
}
