import forge from "node-forge";
import decryptAndVerify from "../smime/decryptAndVerify";
import FDBFactory from "fake-indexeddb/lib/FDBFactory";
import fetchMock from "fetch-mock";
import db from "../pki/SMimePkiDB";
import { readFile } from "./helpers";
import { RevocationResult } from "@bluemind/smime.cacerts.api";
import { base64ToArrayBuffer } from "@bluemind/arraybuffer";
import { MessageBody } from "@bluemind/backend.mail.api";
import * as pki from "../pki";

vi.mock("../../pki/SMimePkiDB");

const mail = {
    value: {
        body: {
            guid: "ea83eb26a5d6ca105c26f0701f17ce87c1edcc23",
            subject: "Chiffré/Signé Outlook",
            smartAttach: false,
            date: 1747149406000,
            created: 1747149406000,
            size: 12314,
            headers: [{ name: "X-BM-Mapi-Version", values: ["5.3.2614"] }],
            recipients: [
                {
                    kind: MessageBody.RecipientKind.Originator,
                    address: "arthur.pendragon@mybm.demo.bluemind.net",
                    dn: "Arthur Pendragon"
                },
                {
                    kind: MessageBody.RecipientKind.Primary,
                    address: "arthur.pendragon@mybm.demo.bluemind.net",
                    dn: "Arthur Pendragon"
                }
            ],
            messageId: "<mapi-1f89c7156b335367b1439a6858beac65@vm-build-5-3.dev.bluemind.net>",
            structure: {
                mime: "application/pkcs7-mime",
                address: "1",
                encoding: "base64",
                size: 12314
            },
            preview: ""
        },
        imapUid: 1
    },
    uid: "1.",
    internalId: 28467914,
    version: 1,
    folderUid: "656a3d42-b565-4be8-97fe-113f111aced4"
};

const privateKey = readFile("privateKeys/arthur_key.pem");

const cert = readFile("certificates/arthur.pem");
const ca = readFile("certificates/ca_arthur.crt");
fetchMock.mock("end:/api/smime_cacerts/smime_cacerts%3Adomain_foo.bar/_all", [{ value: { cert: ca } }], {
    overwriteRoutes: true
});
fetchMock.mock("end:/api/smime_revocation/foo.bar/revoked_clients", [
    {
        status: RevocationResult.RevocationStatus.NOT_REVOKED,
        revocation: { serialNumber: "myCertSerialNumber" }
    }
]);

class MockedKeyAsBlob extends Blob {
    text() {
        return Promise.resolve(JSON.stringify([privateKey]));
    }
}
class MockedSignCertAsBlob extends Blob {
    text() {
        return Promise.resolve(JSON.stringify([cert]));
    }
}

class MockedArturBlob extends Blob {
    arrayBuffer() {
        return Promise.resolve(base64ToArrayBuffer(readFile("parts/outlookEncryptedPart.txt")));
    }
}

const getCertificateSpy = vi.spyOn(pki, "getCertificate");
const arthurCert = forge.pki.certificateFromPem(cert);

getCertificateSpy.mockResolvedValue(arthurCert);

vi.mock("@bluemind/service-worker-utils", () => {
    return {
        fetchRequest: vi.fn(),
        dispatchFetch: vi.fn().mockImplementation(() => ({
            blob: () => new MockedArturBlob(),
            text: () => ""
        })),
        fetchCompleteRequest: vi.fn().mockImplementation(() => "")
    };
});

vi.mock("global", () => {
    return {
        caches: {
            open: vi.fn().mockResolvedValue({
                put: vi.fn().mockResolvedValue(undefined)
            })
        },
        Request: vi.fn().mockImplementation(url => ({ url })),
        Response: vi.fn().mockImplementation(body => ({ body }))
    };
});

describe("one", () => {
    beforeEach(() => {
        global.indexedDB = new FDBFactory();
        fetchMock.mock("/session-infos", {
            login: "mathilde.michau@blue-mind.net",
            sid: "58a1ee1b-0c30-492c-a83f-4396f0a24730",
            defaultEmail: "math@devenv.blue",
            domain: "foo.bar"
        });
        global.Request = vi.fn().mockImplementation(url => ({ url }));
        global.caches = {
            open: vi.fn().mockResolvedValue({
                put: vi.fn().mockResolvedValue(undefined)
            }),
            delete: vi.fn().mockResolvedValue(undefined),
            has: vi.fn().mockResolvedValue(undefined),
            keys: vi.fn().mockResolvedValue(undefined),
            match: vi.fn().mockResolvedValue(undefined)
        };
    }),
        test("decrypt outlook", async () => {
            db.getPrivateKeys = () => Promise.resolve(new MockedKeyAsBlob());
            db.getCertificates = () => Promise.resolve(new MockedSignCertAsBlob());

            const [item] = await decryptAndVerify([mail], "656a3d42-b565-4be8-97fe-113f111aced4");
            expect(item.value.body.headers).toMatchInlineSnapshot(`
              [
                {
                  "name": "X-BM-Mapi-Version",
                  "values": [
                    "5.3.2614",
                  ],
                },
                {
                  "name": "X-MailApp-Encrypted",
                  "values": [
                    "2",
                  ],
                },
              ]
            `);
            expect(item.value.body.structure).toMatchInlineSnapshot(`
              {
                "address": "1",
                "charset": "us-ascii",
                "children": [],
                "contentId": undefined,
                "dispositionType": undefined,
                "encoding": "base64",
                "fileName": "smime.p7m",
                "mime": "application/x-pkcs7-mime",
                "mimeParameters": {
                  "filename": "smime.p7m MIAGCSqGSIb3DQEHAqCAMIACAQExCzAJBgUrDgMCGgUAMIAGCSqGSIb3DQEHAaCAJIAEgggaQ29u dGVudC1UeXBlOiBtdWx0aXBhcnQvYWx0ZXJuYXRpdmU7DQoJYm91bmRhcnk9Ii0tLS09X05leHRQ YXJ0XzAwMF8wMDAwXzAxREJDNDJBLkNFMEY5NUMwIg0KDQpUaGlzIGlzIGEgbXVsdGlwYXJ0IG1l c3NhZ2UgaW4gTUlNRSBmb3JtYXQuDQoNCi0tLS0tLT1fTmV4dFBhcnRfMDAwXzAwMDBfMDFEQkM0 MkEuQ0UwRjk1QzANCkNvbnRlbnQtVHlwZTogdGV4dC9wbGFpbjsNCgljaGFyc2V0PSJpc28tODg1 OS0xIg0KQ29udGVudC1UcmFuc2Zlci1FbmNvZGluZzogOGJpdA0KDQpVbiBtYWlsIGJpZW4gY29t cGxleGWFDQoNCg0KLS0tLS0tPV9OZXh0UGFydF8wMDBfMDAwMF8wMURCQzQyQS5DRTBGOTVDMA0K Q29udGVudC1UeXBlOiB0ZXh0L2h0bWw7DQoJY2hhcnNldD0iaXNvLTg4NTktMSINCkNvbnRlbnQt VHJhbnNmZXItRW5jb2Rpbmc6IHF1b3RlZC1wcmludGFibGUNCg0KPGh0bWwgeG1sbnM6dj0zRCJ1 cm46c2NoZW1hcy1taWNyb3NvZnQtY29tOnZtbCIgPQ0KeG1sbnM6bz0zRCJ1cm46c2NoZW1hcy1t aWNyb3NvZnQtY29tOm9mZmljZTpvZmZpY2UiID0NCnhtbG5zOnc9M0QidXJuOnNjaGVtYXMtbWlj cm9zb2Z0LWNvbTpvZmZpY2U6d29yZCIgPQ0KeG1sbnM6bT0zRCJodHRwOi8vc2NoZW1hcy5taWNy b3NvZnQuY29tL29mZmljZS8yMDA0LzEyL29tbWwiID0NCnhtbG5zPTNEImh0dHA6Ly93d3cudzMu b3JnL1RSL1JFQy1odG1sNDAiPjxoZWFkPjxtZXRhID0NCmh0dHAtZXF1aXY9M0RDb250ZW50LVR5 cGUgY29udGVudD0zRCJ0ZXh0L2h0bWw7ID0NCmNoYXJzZXQ9M0Rpc28tODg1OS0xIj48bWV0YSBu YW1lPTNER2VuZXJhdG9yIGNvbnRlbnQ9M0QiTWljcm9zb2Z0IFdvcmQgPQ0KMTUgKGZpbHRlcmVk IG1lZGl1bSkiPjxzdHlsZT48IS0tDQovKiBGb250IERlZmluaXRpb25zICovDQpAZm9udC1mYWNl DQoJe2ZvbnQtZmFtaWx5OiJDYW1icmlhIE1hdGgiOw0KCXBhbm9zZS0xOjIgNCA1IDMgNSA0IDYg MyAyIDQ7fQ0KQGZvbnQtZmFjZQ0KCXtmb250LWZhbWlseTpBcHRvczt9DQovKiBTdHlsZSBEZWZp bml0aW9ucyAqLw0KcC5Nc29Ob3JtYWwsIGxpLk1zb05vcm1hbCwgZGl2Lk1zb05vcm1hbA0KCXtt YXJnaW46MGNtOw0KCWZvbnQtc2l6ZToxMS4wcHQ7DQoJZm9udC1mYW1pbHk6IkFwdG9zIixzYW5z LXNlcmlmOw0KCW1zby1saWdhdHVyZXM6c3RhbmRhcmRjb250ZXh0dWFsOw0KCW1zby1mYXJlYXN0 LWxhbmd1YWdlOkVOLVVTO30NCnNwYW4uRW1haWxTdHlsZTE3DQoJe21zby1zdHlsZS10eXBlOnBl cnNvbmFsLWNvbXBvc2U7DQoJZm9udC1mYW1pbHk6IkFwdG9zIixzYW5zLXNlcmlmOw0KCWNvbG9y OndpbmRvd3RleHQ7fQ0KLk1zb0NocERlZmF1bHQNCgl7bXNvLXN0eWxlLXR5cGU6ZXhwb3J0LW9u bHk7DQoJZm9udC1zaXplOjExLjBwdDsNCgltc28tZmFyZWFzdC1sYW5ndWFnZTpFTi1VUzt9DQpA cGFnZSBXb3JkU2VjdGlvbjENCgl7c2l6ZTo2MTIuMHB0IDc5Mi4wcHQ7DQoJbWFyZ2luOjcyLjBw dCA3Mi4wcHQgNzIuMHB0IDcyLjBwdDt9DQpkaXYuV29yZFNlY3Rpb24xDQoJe3BhZ2U6V29yZFNl Y3Rpb24xO30NCi0tPjwvc3R5bGU+PCEtLVtpZiBndGUgbXNvIDldPjx4bWw+DQo8bzpzaGFwZWRl ZmF1bHRzIHY6ZXh0PTNEImVkaXQiIHNwaWRtYXg9M0QiMTAyNiIgLz4NCjwveG1sPjwhW2VuZGlm XS0tPjwhLS1baWYgZ3RlIG1zbyA5XT48eG1sPg0KPG86c2hhcGVsYXlvdXQgdjpleHQ9M0QiZWRp dCI+DQo8bzppZG1hcCB2OmV4dD0zRCJlZGl0IiBkYXRhPTNEIjEiIC8+DQo8L286c2hhcGVsYXlv dXQ+PC94bWw+PCFbZW5kaWZdLS0+PC9oZWFkPjxib2R5IGxhbmc9M0RGUiA9DQpsaW5rPTNEIiM0 Njc4ODYiIHZsaW5rPTNEIiM5NjYwN0QiIHN0eWxlPTNEJ3dvcmQtd3JhcDpicmVhay13b3JkJz48 ZGl2ID0NCmNsYXNzPTNEV29yZFNlY3Rpb24xPjxwIGNsYXNzPTNETXNvTm9ybWFsPjxzcGFuIGxh bmc9M0RFTi1VUz5VbiBtYWlsID0NCmJpZW4gY29tcGxleGUmIzgyMzA7PG86cD48L286cD48L3Nw YW4+PC9wPjwvZGl2PjwvYm9keT48L2h0bWw+DQotLS0tLS09X05leHRQYXJ0XzAwMF8wMDAwXzAx REJDNDJBLkNFMEY5NUMwLS0NCgAAAAAAAKCCB8MwggPFMIICraADAgECAhRKydrhHGLqKWkuPWQu a2U4Hf8QkjANBgkqhkiG9w0BAQsFADByMQswCQYDVQQGEwJGUjETMBEGA1UECAwKU29tZS1TdGF0 ZTERMA8GA1UECgwIQmx1ZW1pbmQxDjAMBgNVBAMMBWFkbWluMSswKQYJKoZIhvcNAQkBFhxhZG1p bkBteWJtLmRlbW8uYmx1ZW1pbmQubmV0MB4XDTI1MDUxMzEyMDc0MVoXDTI4MDUxMjEyMDc0MVow cjELMAkGA1UEBhMCRlIxEzARBgNVBAgMClNvbWUtU3RhdGUxETAPBgNVBAoMCEJsdWVtaW5kMQ4w DAYDVQQDDAVhZG1pbjErMCkGCSqGSIb3DQEJARYcYWRtaW5AbXlibS5kZW1vLmJsdWVtaW5kLm5l dDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBALWqp3uTdWxgfsrIJU5FM0Ivp+PZVGlR NRgP7uO/+IVQDOF4MdpN5ypwPPteHJEliJ8kzgPqqz32fuOb1LOPaMZRPr9FXnGttH9AXdc4oN5/ 0dU9pqAjDsBE/ilMFpS54BleZhBKsrUrMp6ALBJjPFQorj+kHQtzE8OeXp4x6RkZiXRC87CuG6ts 3Qrb/HoDReH1HDq/pgvhPx3r/bc1+gOwKRJrLL7V3ZoNRBVuHgm7AuqLR9vmxtsDsSylLuYv6NST Fr2VxiZTAfArkZmXW3gE5ReFNrTBUYsVw/Gsa5glIQ28N4huN1dnCmuGFX86MuNESo7m0KE3zvyM UEEP1TUCAwEAAaNTMFEwHQYDVR0OBBYEFOEqIs/Lw8TC9HUIEEYB5oeS2uBgMB8GA1UdIwQYMBaA FOEqIs/Lw8TC9HUIEEYB5oeS2uBgMA8GA1UdEwEB/wQFMAMBAf8wDQYJKoZIhvcNAQELBQADggEB AAlIpyygq7ZrHhBh1Nz5hpjhi73utjsfeK7bqk5x0xiQFfQ/yCI6tVG7cQE2+h3HBIxKQhIyggjL hYlj9jhCF26aWuQI4si+Jvzkc4r4tvA/AlW9nJ1YrwKfO2tvuuEURJtHmXuTIZjI/dCsfNg5CM77 XaAZRxqVzkpyJXgzkUDLdORKd9UdU8BP26czSHnBFUpc0KjDteYrsUcCjNLW7oLzkpYxUYeXmAol l9/+zufAWuFTlUbQJU+nbi/XezpgGiv/a3ZcpCj/ZhQq69syCqffy9hhmqiXpXQKkioSKePX1kVh UQj7syKcUd9dgkjOtVWiTRiTx8xyR5NtVHMtBv8wggP2MIIC3qADAgECAhRKydrhHGLqKWkuPWQu a2U4Hf8QlTANBgkqhkiG9w0BAQsFADByMQswCQYDVQQGEwJGUjETMBEGA1UECAwKU29tZS1TdGF0 ZTERMA8GA1UECgwIQmx1ZW1pbmQxDjAMBgNVBAMMBWFkbWluMSswKQYJKoZIhvcNAQkBFhxhZG1p bkBteWJtLmRlbW8uYmx1ZW1pbmQubmV0MB4XDTI1MDUxMzE0MTU1NVoXDTI2MDUxMzE0MTU1NVow gagxCzAJBgNVBAYTAkZSMRMwEQYDVQQIDApTb21lLVN0YXRlMREwDwYDVQQHDAhUb3Vsb3VzZTER MA8GA1UECgwIQmx1ZW1pbmQxCzAJBgNVBAsMAkJNMRkwFwYDVQQDDBBBcnRodXIgUGVuZHJhZ29u MTYwNAYJKoZIhvcNAQkBFidhcnRodXIucGVuZHJhZ29uQG15Ym0uZGVtby5ibHVlbWluZC5uZXQw ggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDKhXFT23cLO7D6uOwe5H1hpxo34p+EqzPH bVUy1ZB6r80t0e7l4tQYZhS/TfmbiYSYv+EEVbYruNDtxc1wSqBBxUpIsRNGutnXuAfk8jegWRJY EhQ66tryzdaOg60aGxqRj97s18BP982dqjUGvWo5OGW0c4rZzsudcQbfkjICJfjSZwWHHiblZ/r+ U4SmKj7GU12vsxA81W6lf9LYmgqqyk5qVAe7DksE1VBgyFrbKC95TQLtaHFLiahouiwXZCAPtvhC jz2uMx589lbq4aMf8fYhZbZfXP5qQlex1ay7/hxaxhwFnB8HqfUxk1hkb6dRXomeYvAJBfmN77YS eimxAgMBAAGjTTBLMAkGA1UdEwQCMAAwHQYDVR0OBBYEFHj0GctebzYXGgsOPAKHVZnYINm5MB8G A1UdIwQYMBaAFOEqIs/Lw8TC9HUIEEYB5oeS2uBgMA0GCSqGSIb3DQEBCwUAA4IBAQA2G4cKgJUU PT/0ZQfeIedvJx102Oilw0lVcmReY4ecE2vad8oOsTqXMwZGp4rc9DZPKF//6B2WU6E1ivFvMp8o 9mu6Anzc7C0uV9JQKWucOrfuJrp3jHH5wyq8qRCf7OXv7LhKgTIQjOJ4a6DM7Q+miY9bDD//O50g vYYSFI56TJgrrA+VNBElNTqYHoYgQ6yX3PgpQtdfftpbZNsrraefETN+DPUk8UgTLe39y7ocpjqV sNb3C7Dx3HVR/8HP5GcI3Cct6rAgpWfPpCXIz4aMUrVrf76UThDO9023hwhiaaIVLZVwBUhap/z5 Rx6PLGkaeASrmuy6Jqixn0o3h7kYMYID5zCCA+MCAQEwgYowcjELMAkGA1UEBhMCRlIxEzARBgNV BAgMClNvbWUtU3RhdGUxETAPBgNVBAoMCEJsdWVtaW5kMQ4wDAYDVQQDDAVhZG1pbjErMCkGCSqG SIb3DQEJARYcYWRtaW5AbXlibS5kZW1vLmJsdWVtaW5kLm5ldAIUSsna4Rxi6ilpLj1kLmtlOB3/ EJUwCQYFKw4DAhoFAKCCAjEwGAYJKoZIhvcNAQkDMQsGCSqGSIb3DQEHATAcBgkqhkiG9w0BCQUx DxcNMjUwNTEzMTUxNjQ2WjAjBgkqhkiG9w0BCQQxFgQUN7TAgm+HG6ciQGvxp5AjeN8QsBQwgZMG CSqGSIb3DQEJDzGBhTCBgjALBglghkgBZQMEASowCwYJYIZIAWUDBAEWMAoGCCqGSIb3DQMHMAsG CWCGSAFlAwQBAjAOBggqhkiG9w0DAgICAIAwDQYIKoZIhvcNAwICAUAwBwYFKw4DAhowCwYJYIZI AWUDBAIDMAsGCWCGSAFlAwQCAjALBglghkgBZQMEAgEwgZsGCSsGAQQBgjcQBDGBjTCBijByMQsw CQYDVQQGEwJGUjETMBEGA1UECAwKU29tZS1TdGF0ZTERMA8GA1UECgwIQmx1ZW1pbmQxDjAMBgNV BAMMBWFkbWluMSswKQYJKoZIhvcNAQkBFhxhZG1pbkBteWJtLmRlbW8uYmx1ZW1pbmQubmV0AhRK ydrhHGLqKWkuPWQua2U4Hf8QlTCBnQYLKoZIhvcNAQkQAgsxgY2ggYowcjELMAkGA1UEBhMCRlIx EzARBgNVBAgMClNvbWUtU3RhdGUxETAPBgNVBAoMCEJsdWVtaW5kMQ4wDAYDVQQDDAVhZG1pbjEr MCkGCSqGSIb3DQEJARYcYWRtaW5AbXlibS5kZW1vLmJsdWVtaW5kLm5ldAIUSsna4Rxi6ilpLj1k LmtlOB3/EJUwDQYJKoZIhvcNAQEBBQAEggEANLHMPUmjXbAU9Ce+sJsAHh2ASkMum34RQexkBCFK +RyZGyWxUDP3QYVxaqIkIATXO09DJwQGk0ycfHRODoaKXS1Pq7vKcTVNuAySTRMNyjoGaO1mtNUi YiSt+ZCX34x4x/YC25rSTnFx1TtxepkcF4HMZA+EuXbg5SYELOglOriR9rtlrQWQKdMTLei3fVQk saiFVuVytvXruKGZozC7oc9aAOcHd35Ws+NZSGhqY/QMQ3xUGPVQKUlhnPTAFJslLm0Mh6+dm1F4 GlLDY89VC/5AfrEq0lP7AwYtztTqo9OQzuLTQFdQetT6oKdVpWWx7ytqggs0i8AIJMSrYZk4KAAA AAAAAA==",
                  "name": "smime.p7m",
                  "smime-type": "signed-data Content-Transfer-Encoding: base64 Content-Disposition: attachment",
                },
                "size": 0,
              }
            `);
        });
});
