import { describe, it, expect, vi, beforeEach } from "vitest";

import { ItemValue, OwnerSubscriptionsClient } from "@bluemind/core.container.api";
import ContainerDatasource, { Action } from "@bluemind/service-worker-datasource";

import SubscriptionSyncService from "./SubscriptionSyncService";

vi.mock("@bluemind/core.container.api", () => {
    return {
        OwnerSubscriptionsClient: vi.fn().mockImplementation(() => ({
            getMultiple: vi.fn(),
            changeset: vi.fn(),
            deleteById: vi.fn(),
            updateById: vi.fn(),
            getCompleteById: vi.fn(),
            changesetById: vi.fn(),
            resetData: vi.fn()
        }))
    };
});

vi.mock("@bluemind/service-worker-datasource", () => {
    return {
        default: {
            retrieve: vi.fn()
        },
        Action: {
            CREATE: 0,
            UPDATE: 1,
            DELETE: 2
        }
    };
});
describe("MailboxSyncService", () => {
    let service: SubscriptionSyncService;
    let mockClient: {
        getMultiple: ReturnType<typeof vi.fn>;
        changeset: ReturnType<typeof vi.fn>;
        deleteById: ReturnType<typeof vi.fn>;
        updateById: ReturnType<typeof vi.fn>;
        getCompleteById: ReturnType<typeof vi.fn>;
        changesetById: ReturnType<typeof vi.fn>;
    };
    let mockDB: {
        getItems: ReturnType<typeof vi.fn>;
        getChangeSet: ReturnType<typeof vi.fn>;
        reset: ReturnType<typeof vi.fn>;
    };

    beforeEach(() => {
        vi.clearAllMocks();

        mockDB = {
            getItems: vi.fn(),
            getChangeSet: vi.fn(),
            reset: vi.fn()
        };

        (ContainerDatasource.retrieve as any).mockReturnValue(mockDB);

        service = new SubscriptionSyncService("test-uid", "test-sid", "test-domain");

        mockClient = service.client as unknown as {
            getMultiple: ReturnType<typeof vi.fn>;
            changeset: ReturnType<typeof vi.fn>;
            deleteById: ReturnType<typeof vi.fn>;
            updateById: ReturnType<typeof vi.fn>;
            getCompleteById: ReturnType<typeof vi.fn>;
            changesetById: ReturnType<typeof vi.fn>;
        };
    });

    describe("constructor", () => {
        it("should initialize with the correct parameters", () => {
            expect(OwnerSubscriptionsClient).toHaveBeenCalledWith("test-sid", "test-domain", "test-uid");
            expect(ContainerDatasource.retrieve).toHaveBeenCalledWith("subscription");
            expect(service.client).toBeDefined();
            expect(service.db).toBeDefined();
        });
    });
    describe("getRemoteChangeSet", () => {
        it("should fetch remote changes since a given version", async () => {
            const mockChangeset = {
                created: ["id3", "id4"],
                updated: ["id1", "id2"],
                deleted: ["id5", "id6"],
                version: 1
            };

            mockClient.changeset.mockResolvedValue(mockChangeset);

            const result = await service.getRemoteChangeSet(2);

            expect(mockClient.changeset).toHaveBeenCalledWith(2);
            expect(result).toEqual({
                updated: ["id3", "id4", "id1", "id2"],
                deleted: ["id5", "id6"],
                version: 1
            });
        });
    });

    describe("getRemoteItems", () => {
        it("should fetch remote items", async () => {
            const ids = ["1", "2"];
            const mockResponse1: ItemValue<string>[] = [
                { uid: "id1", value: "id0" },
                { uid: "id2", value: "id2" }
            ];

            mockClient.getMultiple.mockResolvedValueOnce(mockResponse1);

            const result = await service.getRemoteItems(ids);

            expect(mockClient.getMultiple).toHaveBeenCalledTimes(1);
            expect(result).toEqual(mockResponse1);
        });

        it("should ignore null items", async () => {
            const ids = ["1", "2"];
            const mockResponse1: ItemValue<string>[] = [
                { uid: "id1", value: "id0" },
                { uid: "id2", value: "id2" }
            ];

            mockClient.getMultiple.mockResolvedValueOnce([mockResponse1, null]);

            const result = await service.getRemoteItems(ids);

            expect(result).toEqual([mockResponse1]);
        });
    });

    describe("getLocalChangeSet", () => {
        it("should get local changes and categorize them by action type", async () => {
            const mockChangeLog = [
                { action: Action.CREATE, uid: 2 },
                { action: Action.UPDATE, uid: 3 },
                { action: Action.DELETE, uid: 4 },
                { action: Action.DELETE, uid: 5 }
            ];

            mockDB.getChangeSet.mockResolvedValue(mockChangeLog);

            const result = await service.getLocalChangeSet("test-uid");

            expect(mockDB.getChangeSet).toHaveBeenCalled();
            expect(result).toEqual({
                created: [2],
                updated: [3],
                deleted: [4, 5],
                version: 0
            });
        });
    });

    it("Reset DB", async () => {
        await service.resetData("test");

        expect(mockDB.reset).toHaveBeenCalledWith("test");
    });
});
