import { describe, it, expect, vi, beforeEach } from "vitest";

import { MailboxItem, MailboxItemsClient } from "@bluemind/backend.mail.api";
import { ItemValue } from "@bluemind/core.container.api";
import ContainerDatasource, { Action } from "@bluemind/service-worker-datasource";

import MailboxItemSyncService from "./MailboxItemSyncService";

vi.mock("@bluemind/backend.mail.api", () => {
    return {
        MailboxItemsClient: vi.fn().mockImplementation(() => ({
            changeset: vi.fn(),
            filteredChangesetById: vi.fn(),
            deleteById: vi.fn(),
            updateById: vi.fn(),
            getCompleteById: vi.fn(),
            changesetById: vi.fn(),
            multipleGetById: vi.fn(),
            multipleDeleteById: vi.fn(),
            resetData: vi.fn()
        }))
    };
});

vi.mock("@bluemind/service-worker-datasource", () => {
    return {
        default: {
            retrieve: vi.fn()
        },
        Action: {
            CREATE: 0,
            UPDATE: 1,
            DELETE: 2
        }
    };
});

vi.mock("@bluemind/logger", () => {
    return {
        default: {
            log: vi.fn()
        }
    };
});

describe("MailboxItemSyncService", () => {
    let service: MailboxItemSyncService;
    let mockClient: MailboxItemsClient;
    let mockDB: {
        getItems: ReturnType<typeof vi.fn>;
        getChangeSet: ReturnType<typeof vi.fn>;
        commit: ReturnType<typeof vi.fn>;
        reset: ReturnType<typeof vi.fn>;
    };

    beforeEach(() => {
        vi.clearAllMocks();

        mockDB = {
            getItems: vi.fn(),
            getChangeSet: vi.fn(),
            commit: vi.fn(),
            reset: vi.fn()
        };

        (ContainerDatasource.retrieve as any).mockReturnValue(mockDB);

        service = new MailboxItemSyncService("test-uid", "test-sid");

        mockClient = service.client;
    });

    describe("constructor", () => {
        it("should initialize with the correct parameters", () => {
            expect(MailboxItemsClient).toHaveBeenCalledWith("test-sid", "test-uid");
            expect(ContainerDatasource.retrieve).toHaveBeenCalledWith("mail_record");
            expect(service.client).toBeDefined();
            expect(service.db).toBeDefined();
        });
    });
    describe("getRemoteChangeSet", () => {
        it("should fetch remote changes since a given version", async () => {
            const mockChangesetById = {
                created: [{ id: 3 }, { id: 4 }],
                updated: [{ id: 1 }, { id: 2 }],
                deleted: [],
                version: 1
            };

            vi.mocked(mockClient.filteredChangesetById).mockResolvedValue(mockChangesetById);

            const result = await service.getRemoteChangeSet(2);
            expect(mockClient.filteredChangesetById).toHaveBeenCalledWith(2, { must: [], mustNot: ["Deleted"] });
            expect(result).toEqual({
                updated: [3, 4, 1, 2],
                deleted: [],
                version: 1
            });
        });
    });

    describe("getRemoteItems", () => {
        it("should fetch 2 remotes items from 250 ids, done in 2 batch", async () => {
            const generatedIds = Array.from({ length: 250 }, (_, i) => i);
            const mockResponse1: ItemValue<MailboxItem> = { uid: "id1", value: { body: {} } };
            const mockResponse2: ItemValue<MailboxItem> = { uid: "id2", value: { body: {} } };

            vi.mocked(mockClient.multipleGetById).mockResolvedValueOnce([mockResponse1, mockResponse2]);
            vi.mocked(mockClient.multipleGetById).mockResolvedValueOnce([]);

            const result = await service.getRemoteItems(generatedIds);

            expect(mockClient.multipleGetById).toHaveBeenCalledTimes(2);
            expect(result).toEqual([mockResponse1, mockResponse2]);
        });

        it("should handle empty id list", async () => {
            const result = await service.getRemoteItems([]);

            expect(mockClient.getCompleteById).not.toHaveBeenCalled();
            expect(result).toEqual([]);
        });
        it("should ignore null items", async () => {
            const mockResponse1: ItemValue<MailboxItem> = { uid: "id1", value: { body: {} } };
            const mockResponse2: ItemValue<MailboxItem> = null as unknown as ItemValue<MailboxItem>;

            vi.mocked(mockClient.multipleGetById).mockResolvedValueOnce([mockResponse1, mockResponse2]);

            const result = await service.getRemoteItems([0, 1]);

            expect(result).toEqual([mockResponse1]);
        });
    });

    describe("updateRemote", () => {
        it("should update remote with local changes", async () => {
            const changeSet = {
                updated: [1, 2],
                deleted: [3, 4],
                version: 0
            };

            const mockItems = [
                { uid: 1, value: "value1", version: 1 },
                { uid: 2, value: "value2", version: 1 }
            ];

            mockDB.getItems.mockImplementation(ids => {
                return mockItems.filter(item => ids.includes(item.uid));
            });
            vi.mocked(mockClient.updateById).mockResolvedValue({
                version: 1,
                timestamp: 1
            });
            vi.mocked(mockClient.multipleDeleteById).mockResolvedValue();

            await service.updateRemote(changeSet, "test-uid");

            expect(mockDB.getItems).toHaveBeenNthCalledWith(1, [changeSet.updated[0]], "test-uid");
            expect(mockDB.getItems).toHaveBeenNthCalledWith(2, [changeSet.updated[1]], "test-uid");

            expect(mockClient.updateById).toHaveBeenCalledWith(1, "value1");
            expect(mockClient.updateById).toHaveBeenCalledWith(2, "value2");
            expect(mockClient.multipleDeleteById).toHaveBeenCalledWith(changeSet.deleted);
        });
    });

    describe("getLocalChangeSet", () => {
        it("should get local changes and categorize them by action type", async () => {
            const mockChangeLog = [
                { action: Action.UPDATE, uid: 3 },
                { action: Action.DELETE, uid: 4 },
                { action: Action.DELETE, uid: 5 }
            ];

            mockDB.getChangeSet.mockResolvedValue(mockChangeLog);

            const result = await service.getLocalChangeSet("test-uid");

            expect(mockDB.getChangeSet).toHaveBeenCalled();
            expect(result).toEqual({
                created: [],
                updated: [3],
                deleted: [4, 5],
                version: 0
            });
        });
    });

    it("Reset DB", async () => {
        await service.resetData("test");

        expect(mockDB.reset).toHaveBeenCalledWith("test");
    });
});
