import { ConversationFlagUpdate, MailboxItem, MailConversationActionsClient } from "@bluemind/backend.mail.api";
import logger from "@bluemind/logger";
import ContainerDatasource from "@bluemind/service-worker-datasource";

import { relaxedSynchronize } from "../../Synchronization/SyncUtils";

import MailboxItemEntity from "./MailboxItemEntity";

import { MailboxRecordsDB } from "./MailboxRecordsDB";

export default class extends MailConversationActionsClient {
    next?: (...args: Array<unknown>) => Promise<never>;
    async deleteFlag(flagUpdate: ConversationFlagUpdate) {
        if (!flagUpdate.conversationUids || !flagUpdate.mailboxItemFlag) {
            return { version: 0, timestamp: new Date().getTime() };
        }
        try {
            const db = (await ContainerDatasource.retrieve<MailboxItem, number>("mail_record")) as MailboxRecordsDB;
            const flag = flagUpdate.mailboxItemFlag;
            let items = await getConversationsItems(flagUpdate.conversationUids!, this.replicatedMailboxUid, db);
            items = items.map(item => new MailboxItemEntity(item).removeFlag(flag! as string).toItem());

            await db.putItems(items, this.replicatedMailboxUid);
            relaxedSynchronize(this.replicatedMailboxUid, "mail_record");

            const version = (await db.getSyncStatus(this.replicatedMailboxUid))?.version || 0;
            return { version: version, timestamp: new Date().getTime() };
        } catch (error) {
            logger.error(
                `[MailboxItemLocalDBProxy] Failed to deleteFlag for uid ${this.replicatedMailboxUid}, ignore PROXY`,
                error
            );
        }

        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
        return this.next!();
    }

    async addFlag(flagUpdate: ConversationFlagUpdate) {
        if (!flagUpdate.conversationUids || !flagUpdate.mailboxItemFlag) {
            return { version: 0, timestamp: new Date().getTime() };
        }
        try {
            const db = (await ContainerDatasource.retrieve<MailboxItem, number>("mail_record")) as MailboxRecordsDB;
            const flag = flagUpdate.mailboxItemFlag;

            let items = await getConversationsItems(flagUpdate.conversationUids!, this.replicatedMailboxUid, db);
            items = items.map(item => new MailboxItemEntity(item).addFlag(flag! as string).toItem());

            await db.putItems(items, this.replicatedMailboxUid);
            relaxedSynchronize(this.replicatedMailboxUid, "mail_record");

            const version = (await db.getSyncStatus(this.replicatedMailboxUid))?.version || 0;
            return { version: version, timestamp: new Date().getTime() };
        } catch (error) {
            logger.error(
                `[MailboxItemLocalDBProxy] Failed to addFlag for uid ${this.replicatedMailboxUid}, ignore PROXY`,
                error
            );
        }

        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
        return this.next!();
    }

    async multipleDeleteById(conversationUids: string[] = []): Promise<void> {
        try {
            const db = (await ContainerDatasource.retrieve<MailboxItem, number>("mail_record")) as MailboxRecordsDB;
            await db.deleteConversations(conversationUids, this.replicatedMailboxUid);
            relaxedSynchronize(this.replicatedMailboxUid, "mail_record");
            return;
        } catch (error) {
            logger.error(
                `[ConversationActionLocalDBProxy] Failed to multipleDeleteById for container uid ${this.replicatedMailboxUid} and uids ${conversationUids}, ignore PROXY`,
                error
            );
        }
        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
        return this.next!();
    }
}

async function getConversationsItems(conversationUids: string[], containerUid: string, db: MailboxRecordsDB) {
    const conversations = await db.getConversationMessageRefs(conversationUids, containerUid);
    const allItemIds = conversations.flatMap(({ messageRefs }) => messageRefs?.map(({ itemId }) => itemId!) || []);
    return db.getItems(allItemIds, containerUid);
}
