import serviceWorker from "@bluemind/commons.light/utils/service-worker";
import type { ContainerSubscriptionModel } from "@bluemind/core.container.api";
import logger from "@bluemind/logger";
import ContainerDatasource from "@bluemind/service-worker-datasource";
import session from "@bluemind/session";

import type { MessageHandler } from "../ MessageHandlerRegistry";

import db, { type SystemKey } from "../db/EnvironmentDB";

import { synchronize } from "./Synchronize";
import SynchronizeQueue from "./SynchronizeQueue";
import { isLocal } from "./SyncUtils";

declare const self: ServiceWorkerGlobalScope;

export const SynchronizeMessageHandler: MessageHandler<
    "SYNCHRONIZE",
    { uid: string; type: string; forceSync: boolean }
> = {
    type: "SYNCHRONIZE",
    async callback(body) {
        if ((await isLocal(body.uid, body.type)) || body.forceSync) {
            SynchronizeQueue.queue(body);
        } else {
            serviceWorker.postMessage("UPDATED", body);
        }
    }
};

const ACTIVE_CLIENTS_KEY: SystemKey<string[]> = "activeClients";
export const ConnectedMessageHandler: MessageHandler<"CONNECTED"> = {
    type: "CONNECTED",
    async callback(body, headers, event) {
        const clients = (await self.clients.matchAll()).reduce((set, client) => set.add(client.id), new Set<string>());

        const activeClients = new Set(
            ((await db.getSystemValue(ACTIVE_CLIENTS_KEY)) || []).filter(client => clients.has(client))
        );
        try {
            if (activeClients.size === 0) {
                synchronizeFull();
            }
            activeClients.add((event.source as Client).id);
            await db.setSystemValue(ACTIVE_CLIENTS_KEY, [...activeClients]);
        } catch (error) {
            logger.error("[Synchronization] Fail to synchronizeFull", error);
        }
    }
};
export const DisconnectMessageHandler: MessageHandler<"DISCONNECTED"> = {
    type: "DISCONNECTED",
    async callback(body, headers, event) {
        const activeClients = new Set((await db.getSystemValue(ACTIVE_CLIENTS_KEY)) || []);
        activeClients.delete((event.source as Client).id);
        await db.setSystemValue(ACTIVE_CLIENTS_KEY, [...activeClients]);
    }
};

async function synchronizeFull() {
    logger.log("[Synchronization] synchronize all subscribed containers");
    const uid = `${session.userId}@${session.domain}.subscriptions`;
    await synchronize({ uid, type: "owner_subscriptions" });
    const subscriptions = await (
        await ContainerDatasource.retrieve<ContainerSubscriptionModel, string>("owner_subscriptions")
    ).getAllItems(uid);
    for (const sub of subscriptions) {
        SynchronizeQueue.queue({
            uid: sub.value.containerUid as string,
            type: sub.value.containerType as string
        });
    }
}
