import { AddressBookClient, VCard } from "@bluemind/addressbook.api";
import { ChangeSet, SyncService } from "@bluemind/commons.light/model/synchronization";
import { chunk } from "@bluemind/commons.light/utils/array";
import { ContainerChangeset, ItemValue } from "@bluemind/core.container.api";
import ContainerDatasource, { type ContainerDB } from "@bluemind/service-worker-datasource";

export default class AddressbookSyncService implements SyncService<VCard, string> {
    client: AddressBookClient;
    db: Promise<ContainerDB<VCard, string>>;
    constructor(uid: string, sid: string) {
        this.client = new AddressBookClient(sid, uid);
        this.db = ContainerDatasource.retrieve<VCard, string>("addressbook");
    }
    async getRemoteChangeSet(since: number) {
        const { created, updated, deleted, version } = (await this.client.changeset(since)) as Required<
            ContainerChangeset<string>
        >;

        const updatedIds = created.reverse().concat(updated.reverse());

        return { updated: updatedIds, deleted, version };
    }
    async getRemoteItems(ids: string[]) {
        const chunks = [...chunk(ids, 200)];
        const responses = await Promise.all(chunks.map(chunk => this.client.multipleGet(chunk)));
        const result: ItemValue<VCard>[] = [];
        return result.concat(...responses).filter(Boolean);
    }
    async updateRemote(changeSet: ChangeSet<string>, containerUid: string) {
        const itemsAdded = await (await this.db).getItems(changeSet.updated, containerUid);
        const itemsUpdated = await (await this.db).getItems(changeSet.updated, containerUid);
        const items = {
            added: itemsAdded.map(item => ({
                uid: item.uid,
                value: item.value
            })),
            modify: itemsUpdated.map(item => ({
                uid: item.uid,
                value: item.value
            })),
            delete: changeSet.deleted.map(item => ({
                uid: item
            }))
        };
        const { added = [], updated = [], removed = [] } = await this.client.updates(items);
        await (await this.db).commit(added.concat(updated).concat(removed), containerUid);
    }

    async getLocalChangeSet(containerUid: string) {
        const changeLog = await (await this.db).getChangeSet(containerUid);
        const { created, updated, deleted } = changeLog.reduce<{
            created: string[];
            updated: string[];
            deleted: string[];
        }>(
            (result, item) => {
                const { action, uid } = item;
                switch (action) {
                    case 0:
                        result.created?.push(uid);
                        break;
                    case 1:
                        result.updated?.push(uid);
                        break;
                    case 2:
                        result.deleted?.push(uid);
                        break;
                }

                return result;
            },
            { created: [], updated: [], deleted: [] }
        );
        return { created, updated, deleted, version: 0 };
    }

    async resetData(containerUid: string) {
        await (await this.db).reset(containerUid);
    }
}
