/* eslint-disable @typescript-eslint/no-non-null-assertion */
import { AddressBookClient, type VCard } from "@bluemind/addressbook.api";
import type { ItemValue } from "@bluemind/core.container.api";
import logger from "@bluemind/logger";
import ContainerDatasource from "@bluemind/service-worker-datasource";

import { isLocal, isSynchronized, relaxedSynchronize, nudgeSync } from "../../Synchronization/SyncUtils";

class NoContentError extends Error {
    constructor(message = "") {
        super(message);
        this.name = "NoContentError";
    }
}

export default class extends AddressBookClient {
    next?: (...args: Array<unknown>) => Promise<never>;
    async update(uid: string, card: VCard) {
        try {
            sanitizeVCard(card);
            const db = await ContainerDatasource.retrieve<VCard, string>("addressbook");
            if (db) {
                await db.putItems([{ uid, value: card }], this.containerUid);
                relaxedSynchronize(this.containerUid, "addressbook");
                return;
            }
        } catch (error) {
            logger.error(`[AddressBookLocalDBProxy] Failed to update uid ${uid}, ignore PROXY`, error);
        }

        return this.next!();
    }
    async delete(uid: string) {
        try {
            const db = await ContainerDatasource.retrieve<VCard, string>("addressbook");
            await db.deleteItems([uid], this.containerUid);
            relaxedSynchronize(this.containerUid, "addressbook");
            return;
        } catch (error) {
            logger.error(`[AddressBookLocalDBProxy] Failed to delete uid ${uid}, ignore PROXY`, error);
        }

        return this.next!();
    }
    async getComplete(uid: string): Promise<ItemValue<VCard>> {
        try {
            if (!(await isLocal(this.containerUid, "addressbook"))) {
                return this.next!();
            }

            if (!isSynchronized(this.containerUid, "addressbook")) {
                await nudgeSync(this.containerUid, "addressbook");
            }

            const db = await ContainerDatasource.retrieve<VCard, string>("addressbook");
            const result = await db.getItems([uid], this.containerUid);
            if (result.length > 0) {
                return result[0];
            } else {
                throw new NoContentError();
            }
        } catch (error) {
            logger.error(
                `[AddressBookLocalDBProxy] Failed to get item from local database uid ${uid}, ignore PROXY`,
                error
            );
        }

        return this.next!();
    }
    async getCompleteById(id: number): Promise<ItemValue<VCard>> {
        try {
            if (!(await isLocal(this.containerUid, "addressbook"))) {
                return this.next!();
            }

            if (!isSynchronized(this.containerUid, "addressbook")) {
                await nudgeSync(this.containerUid, "addressbook");
            }

            const db = await ContainerDatasource.retrieve<VCard, string>("addressbook");
            const result = (await db.getAllItems(this.containerUid)).filter(
                item => item.internalId && item.internalId === id
            );
            if (result.length > 0) {
                return result[0];
            } else {
                throw new NoContentError();
            }
        } catch (error) {
            logger.error(
                `[AddressBookLocalDBProxy] Failed to get item from local database for id ${id}, ignore PROXY`,
                error
            );
        }

        return this.next!();
    }
    async multipleGet(uids: Array<string>): Promise<Array<ItemValue<VCard>>> {
        try {
            if (!(await isLocal(this.containerUid, "addressbook"))) {
                return this.next!();
            }

            if (!isSynchronized(this.containerUid, "addressbook")) {
                await nudgeSync(this.containerUid, "addressbook");
            }

            const db = await ContainerDatasource.retrieve<VCard, string>("addressbook");
            return await db.getItems(uids, this.containerUid);
        } catch (error) {
            logger.error(`[AddressBookLocalDBProxy] Failed to get items from local database, ignore PROXY`, error);
        }

        return this.next!();
    }

    async multipleGetById(ids: number[]) {
        try {
            if (!(await isLocal(this.containerUid, "addressbook"))) {
                return this.next!();
            }

            if (!isSynchronized(this.containerUid, "addressbook")) {
                await nudgeSync(this.containerUid, "addressbook");
            }

            const db = await ContainerDatasource.retrieve<VCard, string>("addressbook");
            return (await db.getAllItems(this.containerUid)).filter(
                item => item.internalId && ids.includes(item.internalId)
            );
        } catch (error) {
            logger.error(`[AddressBookLocalDBProxy] Failed to get items from local database, ignore PROXY`, error);
        }

        return this.next!();
    }
}

function sanitizeVCard(vcard: VCard): VCard {
    vcard.kind = vcard.kind || "individual";
    return vcard;
}
