import { describe, it, expect, vi, beforeEach, afterEach } from "vitest";

import { ContainerSyncOptions } from "@bluemind/commons.light/model/synchronization";

import { synchronize } from "./Synchronize";
import SynchronizeQueue from "./SynchronizeQueue";
vi.mock("./Synchronize", () => ({
    synchronize: vi.fn()
}));

describe("SynchronizeQueue", () => {
    beforeEach(() => {
        vi.clearAllMocks();
        vi.mocked(synchronize).mockResolvedValue(true);
        SynchronizeQueue.fifo = [];
        SynchronizeQueue.inProgress = new Map();
    });

    afterEach(() => {
        vi.resetAllMocks();
    });

    describe("add", () => {
        it("add a new container to the queue when it does not already exist", () => {
            const options: ContainerSyncOptions = { uid: "123", type: "test" };

            SynchronizeQueue.add(options);

            expect(SynchronizeQueue.fifo).toHaveLength(1);
            expect(SynchronizeQueue.fifo[0]).toEqual({ uid: "123", type: "test", priority: 0 });
        });

        it("Doesn't add a container if it already exists with higher priority", () => {
            const highPriorityOptions: ContainerSyncOptions = { uid: "123", type: "test", priority: 10 };
            const lowPriorityOptions: ContainerSyncOptions = { uid: "123", type: "test", priority: 5 };

            SynchronizeQueue.add(highPriorityOptions);
            SynchronizeQueue.add(lowPriorityOptions);

            expect(SynchronizeQueue.fifo).toHaveLength(1);
            expect(SynchronizeQueue.fifo[0].priority).toBe(10);
        });

        it("Update priority if existing container has lower priority", () => {
            const lowPriorityOptions: ContainerSyncOptions = { uid: "123", type: "test", priority: 5 };
            const highPriorityOptions: ContainerSyncOptions = { uid: "123", type: "test", priority: 10 };

            SynchronizeQueue.add(lowPriorityOptions);
            SynchronizeQueue.add(highPriorityOptions);

            expect(SynchronizeQueue.fifo).toHaveLength(1);
            expect(SynchronizeQueue.fifo[0].priority).toBe(10);
        });

        it("Sort the queue by priority (highest first)", () => {
            const options1: ContainerSyncOptions = { uid: "123", type: "test", priority: 5 };
            const options2: ContainerSyncOptions = { uid: "456", type: "test", priority: 10 };
            const options3: ContainerSyncOptions = { uid: "789", type: "test", priority: 3 };

            SynchronizeQueue.add(options1);
            SynchronizeQueue.add(options2);
            SynchronizeQueue.add(options3);

            expect(SynchronizeQueue.fifo).toHaveLength(3);
            expect(SynchronizeQueue.fifo[0].uid).toBe("456");
            expect(SynchronizeQueue.fifo[1].uid).toBe("123");
            expect(SynchronizeQueue.fifo[2].uid).toBe("789");
        });
    });

    describe("flush", () => {
        it("Process all items in the queue", async () => {
            const options1: ContainerSyncOptions = { uid: "123", type: "test", priority: 5 };
            const options2: ContainerSyncOptions = { uid: "456", type: "test", priority: 10 };

            SynchronizeQueue.add(options1);
            SynchronizeQueue.add(options2);

            await SynchronizeQueue.flush();

            expect(SynchronizeQueue.fifo).toHaveLength(0);
            expect(synchronize).toHaveBeenCalledTimes(2);
            expect(synchronize).toHaveBeenNthCalledWith(1, { uid: "456", type: "test", priority: 10 });
            expect(synchronize).toHaveBeenNthCalledWith(2, { uid: "123", type: "test", priority: 5 });
        });

        it("Doesn't start a new flush if one is already in progress", async () => {
            SynchronizeQueue.inProgress.set("123", new Promise(() => true));

            const result = await SynchronizeQueue.flush();

            expect(result).toBe(false);
            expect(synchronize).not.toHaveBeenCalled();
        });

        it("Reset flushInProgress after completion", async () => {
            const options: ContainerSyncOptions = { uid: "123", type: "test" };

            SynchronizeQueue.add(options);
            await SynchronizeQueue.flush();

            expect(synchronize).toHaveBeenNthCalledWith(1, { uid: "123", type: "test", priority: 0 });
        });
    });

    describe("queue", () => {
        it("Add options to the queue and flush it", async () => {
            const options: ContainerSyncOptions = { uid: "123", type: "test" };

            const addSpy = vi.spyOn(SynchronizeQueue, "add");
            const flushSpy = vi.spyOn(SynchronizeQueue, "flush");

            await SynchronizeQueue.queue(options);

            expect(addSpy).toHaveBeenCalledWith(options);
            expect(flushSpy).toHaveBeenCalled();
        });
    });
    describe("flushOne", () => {
        it("return promise synchronize if the container is in progress", async () => {
            const options: ContainerSyncOptions = { uid: "963", type: "test", priority: 5 };
            SynchronizeQueue.queue(options);
            // reset synchronize call count
            vi.mocked(synchronize).mockClear();

            const result = SynchronizeQueue.flushOne(options);

            expect(result).toBeInstanceOf(Promise);
            await expect(result).resolves.toBeTruthy();
            expect(synchronize).not.toHaveBeenCalled();
        });
        it("return nothing if the container is not in queue", async () => {
            const options: ContainerSyncOptions = { uid: "963", type: "test", priority: 5 };

            const result = SynchronizeQueue.flushOne(options);

            expect(result).toBeInstanceOf(Promise);
            await expect(result).resolves.toBeUndefined();
            expect(synchronize).not.toHaveBeenCalled();
        });
        it("return promise synchronize if the container is in queue", async () => {
            const options: ContainerSyncOptions = { uid: "963", type: "test", priority: 5 };
            SynchronizeQueue.add(options);

            const result = SynchronizeQueue.flushOne(options);

            expect(result).toBeInstanceOf(Promise);
            await expect(result).resolves.toBeTruthy();
            expect(synchronize).toHaveBeenCalled();
        });
    });

    describe("find", () => {
        it("Find a container in the queue by uid", () => {
            const options1: ContainerSyncOptions = { uid: "123", type: "test", priority: 5 };
            const options2: ContainerSyncOptions = { uid: "456", type: "test", priority: 10 };

            SynchronizeQueue.add(options1);
            SynchronizeQueue.add(options2);

            const searchOptions: ContainerSyncOptions = { uid: "123", type: "test" };
            const result = SynchronizeQueue.fifo.find(({ uid }) => searchOptions.uid === uid);

            expect(result).toBeDefined();
            expect(result?.uid).toBe("123");
            expect(result?.priority).toBe(5);
        });

        it("Search return undefined if no container found", () => {
            const options: ContainerSyncOptions = { uid: "123", type: "test" };

            SynchronizeQueue.add(options);

            const searchOptions: ContainerSyncOptions = { uid: "456", type: "test" };
            const result = SynchronizeQueue.fifo.find(({ uid }) => searchOptions.uid === uid);

            expect(result).toBeUndefined();
        });
    });
});
