import pLimit from "p-limit";
import { Limit } from "p-limit";

import type {
    ContainerSyncOptions,
    ItemIdType,
    SyncService,
    SyncStatus
} from "@bluemind/commons.light/model/synchronization";
import serviceWorker, { defaultSyncStatus } from "@bluemind/commons.light/utils/service-worker";
import logger from "@bluemind/logger";
import ContainerDatasource, { type ContainerDB } from "@bluemind/service-worker-datasource";

import { getLock, isLocked, unlock } from "../Instance";

import SynchronizeQueue from "./SynchronizeQueue";
import SyncServiceProvider from "./SyncServiceProvider";

export async function synchronize(container: ContainerSyncOptions): Promise<boolean> {
    if (container.uid.endsWith("@global.virt.subscriptions")) {
        return false;
    }
    logger.log(`[Synchronization] synchronize ${container.uid}`);
    try {
        const syncServiceClient = await SyncServiceProvider.get(container.uid, container.type);

        if (syncServiceClient === undefined) {
            logger.log(`[Synchronization] ${container.type} has no SyncProvider, ${container.uid} will not be synced`);
            return false;
        }
        const db = await ContainerDatasource.retrieve(container.type);

        await flagAsStale(container.uid, db);
        const updated = await limit(container.uid, async () => {
            const { stale, version, lock } = (await db.getSyncStatus(container.uid)) as SyncStatus;
            if (stale && !isLocked(lock)) {
                await flagAsLocked(container.uid, db);
                const newVersion = await synchronizeContainerToVersion(syncServiceClient, db, container, version);
                await flagAsUnlocked(container.uid, db, newVersion);
                return newVersion !== version;
            }
            return false;
        });
        if (updated) {
            serviceWorker.postMessage("UPDATED", container);
        }
        return true;
    } catch (error) {
        logger.error(`[Synchronization] Fail to synchronize ${container?.uid}`, error);
        return false;
    }
}

async function synchronizeContainerToVersion<T, K extends ItemIdType>(
    syncServiceClient: SyncService<T, K>, // FIXME : The identifier type must be review
    db: ContainerDB<T, K>, // FIXME : The identifier type must be review
    { uid, type }: ContainerSyncOptions,
    version: number
): Promise<number> {
    try {
        const localChangeSet = await syncServiceClient.getLocalChangeSet(uid);
        if (hasLocalChange(localChangeSet)) {
            await syncServiceClient.updateRemote(localChangeSet, uid);
        }

        const changeSet = await syncServiceClient.getRemoteChangeSet(version);
        flagAsIdle(uid, db);
        if (!changeSet || changeSet.version === version) {
            return version;
        }
        if (changeSet.version > version) {
            const itemsUpdated = await syncServiceClient.getRemoteItems(changeSet.updated);
            await db.putItemsAndCommit(itemsUpdated, uid);
            await db.deleteItemsAndCommit(changeSet.deleted, uid);

            logger.log(`[Synchronization] ${uid} has been synced from ${version} to ${changeSet.version}`);
            return changeSet?.version;
        } else {
            logger.warn(`[Synchronization] The local version is greater than the remote, ${uid}'s data will be reset`);
            await syncServiceClient.resetData(uid);
            SynchronizeQueue.queue({ uid, type });
            return 0;
        }
    } catch (error) {
        logger.error(`[Synchronization] error while syncing changeset for ${uid}`, error);
        return version;
    }
}

const limits: { [uid: string]: Limit } = {};
export function limit<T>(uid: string, fn: () => Promise<T>): Promise<T> {
    if (!(uid in limits)) {
        limits[uid] = pLimit(1);
    }
    return limits[uid](fn);
}

async function getSyncStatus<T, K extends ItemIdType>(containerUid: string, db: ContainerDB<T, K>) {
    const status = (await db.getSyncStatus(containerUid)) || defaultSyncStatus;
    return status;
}
async function flagAsLocked<T, K extends ItemIdType>(containerUid: string, db: ContainerDB<T, K>) {
    const syncStatus = await getSyncStatus(containerUid, db);
    syncStatus.lock = getLock();
    await db.setSyncStatus(syncStatus, containerUid);
}

async function flagAsUnlocked<T, K extends ItemIdType>(containerUid: string, db: ContainerDB<T, K>, version: number) {
    const syncStatus = await getSyncStatus(containerUid, db);
    syncStatus.version = version;
    syncStatus.lock = unlock();
    await db.setSyncStatus(syncStatus, containerUid);
}

async function flagAsStale<T, K extends ItemIdType>(
    containerUid: string,
    db: ContainerDB<T, K>,
    updatedVersion?: number
) {
    const syncStatus = await getSyncStatus(containerUid, db);
    if (updatedVersion) {
        syncStatus.version = updatedVersion;
    }
    syncStatus.stale = true;
    await db.setSyncStatus(syncStatus, containerUid);
}

async function flagAsIdle<T, K extends ItemIdType>(containerUid: string, db: ContainerDB<T, K>) {
    const syncStatus = await getSyncStatus(containerUid, db);
    syncStatus.stale = false;
    await db.setSyncStatus(syncStatus, containerUid);
}

function hasLocalChange(changeSet: { updated: (string | number)[]; deleted: (string | number)[] }) {
    return changeSet.updated.length > 0 || changeSet.deleted.length > 0;
}
