/* eslint-disable @typescript-eslint/no-non-null-assertion */
import { type MailboxItem, MailboxItemsClient } from "@bluemind/backend.mail.api";
import { sortBy } from "@bluemind/commons.light/utils/collection";
import { isDefined } from "@bluemind/commons.light/utils/lang";
import type { ItemFlag, ItemFlagFilter, SortDescriptor } from "@bluemind/core.container.api";
import logger from "@bluemind/logger";
import ContainerDatasource from "@bluemind/service-worker-datasource";

import { nudgeSync, isLocal, isSynchronized } from "../../Synchronization/SyncUtils";

import type { MailboxRecordsDB, MailItemLight } from "./MailboxRecordsDB";

export default class extends MailboxItemsClient {
    next?: (...args: Array<unknown>) => Promise<never>;
    async count(filter: ItemFlagFilter) {
        try {
            if (!(await isLocal(this.replicatedMailboxUid, "mail_record", filter))) {
                return this.next!();
            }

            if (!isSynchronized(this.replicatedMailboxUid, "mail_record")) {
                await nudgeSync(this.replicatedMailboxUid, "mail_record");
            }

            const db = (await ContainerDatasource.retrieve<MailboxItem, number>("mail_record")) as MailboxRecordsDB;
            const allMailItems = await db.getAllItemLight(this.replicatedMailboxUid);
            const total = allMailItems.filter(item => filterByFlags(filter, item.flags)).length;
            return { total };
        } catch (error) {
            logger.error(
                `[MailboxItemLocalDBProxy] Failed to count for uid ${this.replicatedMailboxUid}, ignore PROXY`,
                error
            );
        }
        return this.next!();
    }

    async multipleGetById(ids: number[]) {
        try {
            if (!(await isLocal(this.replicatedMailboxUid, "mail_record"))) {
                return this.next!();
            }

            if (!isSynchronized(this.replicatedMailboxUid, "mail_record")) {
                await nudgeSync(this.replicatedMailboxUid, "mail_record");
            }

            const db = (await ContainerDatasource.retrieve<MailboxItem, number>("mail_record")) as MailboxRecordsDB;
            const items = (await db.getItems(ids, this.replicatedMailboxUid)).filter(isDefined);
            if (items.length > 0 || ids.length === 0) {
                return items;
            }
        } catch (error) {
            logger.error(
                `[MailboxItemLocalDBProxy] Failed to multipleGetById for uid ${this.replicatedMailboxUid}, ignore PROXY`,
                error
            );
        }
        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
        return this.next!();
    }

    async sortedIds(sort?: SortDescriptor) {
        try {
            if (!(await isLocal(this.replicatedMailboxUid, "mail_record", sort?.filter))) {
                return this.next!();
            }

            if (!isSynchronized(this.replicatedMailboxUid, "mail_record")) {
                await nudgeSync(this.replicatedMailboxUid, "mail_record");
            }

            const db = (await ContainerDatasource.retrieve<MailboxItem, number>("mail_record")) as MailboxRecordsDB;
            const allMailItems: Array<MailItemLight> = await db.getAllItemLight(this.replicatedMailboxUid);
            const iteratee = getIteratee(sort?.fields?.at(0));
            const sorted = sortBy(allMailItems, iteratee);
            const ids: number[] = [];
            sorted.forEach(({ internalId, flags }) => {
                if (matchFilter(flags, sort?.filter)) {
                    ids.push(internalId);
                }
            });
            return sort?.fields?.at(0)?.dir === "Desc" ? ids.reverse() : ids;
        } catch (error) {
            logger.error(
                `[MailboxItemLocalDBProxy] Failed to sortedIds for uid ${this.replicatedMailboxUid}, ignore PROXY`,
                error
            );
        }
        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
        return this.next!();
    }

    //FIXME : ITINERIS
    /* async addFlag(flagUpdate?: FlagUpdate) {
        if (!flagUpdate || !flagUpdate.itemsId || !flagUpdate.mailboxItemFlag) {
            return { version: 0, timestamp: new Date().getTime() };
        }
        try {
            const db = (await ContainerDatasource.retrieve<MailboxItem, number>("mail_record")) as MailboxRecordsDB;
            const flag = flagUpdate!.mailboxItemFlag!;
            let items = await db.getItems(flagUpdate!.itemsId!, this.replicatedMailboxUid);

            items = items.map(item => new MailboxItemEntity(item).addFlag(flag as string).toItem());
            await db.putItems(items, this.replicatedMailboxUid);
            relaxedSynchronize(this.replicatedMailboxUid, "mail_record");

            const version = (await db.getSyncStatus(this.replicatedMailboxUid))?.version || 0;
            return { version: version, timestamp: new Date().getTime() };
        } catch (error) {
            logger.error(
                `[MailboxItemLocalDBProxy] Failed to addFlag for uid ${this.replicatedMailboxUid}, ignore PROXY`,
                error
            );
        }

        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
        return this.next!();
    } */

    //FIXME : ITINERIS
    /* async deleteFlag(flagUpdate?: FlagUpdate) {
        if (!flagUpdate || !flagUpdate.itemsId || !flagUpdate.mailboxItemFlag) {
            return { version: 0, timestamp: new Date().getTime() };
        }
        try {
            const flag = flagUpdate!.mailboxItemFlag!;

            const db = (await ContainerDatasource.retrieve<MailboxItem, number>("mail_record")) as MailboxRecordsDB;
            let items = await db.getItems(flagUpdate!.itemsId!, this.replicatedMailboxUid);
            items = items.map(item => new MailboxItemEntity(item).removeFlag(flag as string).toItem());

            await db.putItems(items, this.replicatedMailboxUid);
            relaxedSynchronize(this.replicatedMailboxUid, "mail_record");

            const version = (await db.getSyncStatus(this.replicatedMailboxUid))?.version || 0;
            return { version: version, timestamp: new Date().getTime() };
        } catch (error) {
            logger.error(
                `[MailboxItemLocalDBProxy] Failed to deleteFlag for uid ${this.replicatedMailboxUid} and ids ${flagUpdate?.itemsId}, ignore PROXY`,
                error
            );
        }
        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
        return this.next!();
    } */

    //FIXME : ITINERIS
    /* async deleteById(id: number): Promise<void> {
        try {
            const db = (await ContainerDatasource.retrieve<MailboxItem, number>("mail_record")) as MailboxRecordsDB;
            db.deleteItems([id], this.replicatedMailboxUid);
            relaxedSynchronize(this.replicatedMailboxUid, "mail_record");
            return;
        } catch (error) {
            logger.error(
                `[MailboxItemLocalDBProxy] Failed to delete for uid ${this.replicatedMailboxUid} and id ${id}, ignore PROXY`,
                error
            );
        }
        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
        return this.next!();
    } */

    //FIXME : ITINERIS
    /* async multipleDeleteById(ids?: Array<number>, bypassDeletedItems?: boolean): Promise<void> {
        try {
            const db = (await ContainerDatasource.retrieve<MailboxItem, number>("mail_record")) as MailboxRecordsDB;
            db.deleteItems(ids || [], this.replicatedMailboxUid);
            relaxedSynchronize(this.replicatedMailboxUid, "mail_record");
            return;
        } catch (error) {
            logger.error(
                `[MailboxItemLocalDBProxy] Failed to multipleDeleteById for container uid ${this.replicatedMailboxUid} and ids ${ids}, ignore PROXY`,
                error
            );
        }
        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
        return this.next!();
    } */
}

export function filterByFlags(expected: ItemFlagFilter | undefined, flags: ItemFlag[]) {
    return (
        expected?.must?.every(flag => flags.includes(flag)) && !expected?.mustNot?.some(flag => flags.includes(flag))
    );
}

function getIteratee(field?: SortDescriptor.Field) {
    switch (field?.column) {
        case "date":
        case "size":
        case "subject":
        case "sender":
            return field.column;
        case "internal_date":
        default:
            return "date";
    }
}
function matchFilter(flags: Array<ItemFlag>, filter?: ItemFlagFilter) {
    return filter?.must?.every(flag => flags.includes(flag)) && !filter.mustNot?.some(flag => flags.includes(flag));
}
