/* eslint-disable @typescript-eslint/no-non-null-assertion */
import {
    type Conversation,
    type MailboxFolder,
    type MailboxItem,
    MailConversationClient
} from "@bluemind/backend.mail.api";

import { sortBy } from "@bluemind/commons.light/utils/collection";
import { ItemFlag, type ItemFlagFilter, type SortDescriptor } from "@bluemind/core.container.api";
import logger from "@bluemind/logger";
import ContainerDatasource from "@bluemind/service-worker-datasource";

import { isLocal, isSynchronized, nudgeSync } from "../../Synchronization/SyncUtils";

import type { MailboxRecordsDB } from "./MailboxRecordsDB";

export default class extends MailConversationClient {
    next?: (...args: Array<unknown>) => Promise<never>;

    mailbox: string;
    constructor(apiKey: string, subtreeContainer: string, base?: string) {
        super(apiKey, subtreeContainer, base);
        this.mailbox = "mailbox:acls-" + this.subtreeContainer?.split("!")[1].replace("user.", "");
    }

    async multipleGet(uids: Array<string>): Promise<Array<Conversation>> {
        try {
            if (!(await isLocal(this.mailbox, "mailboxacl"))) {
                return this.next!();
            }

            if (!isSynchronized(this.mailbox, "mailboxacl")) {
                await nudgeSync(this.mailbox, "mailboxacl");
            }
            return getConversations(uids, this.mailbox);
        } catch (error) {
            logger.error(`[ConversationLocalDBProxy] Failed to multiple get conversations, ignore PROXY`, uids, error);
        }

        return this.next!();
    }
    async get(uid: string) {
        try {
            if (!(await isLocal(this.mailbox, "mailboxacl"))) {
                return this.next!();
            }

            if (!isSynchronized(this.mailbox, "mailboxacl")) {
                await nudgeSync(this.mailbox, "mailboxacl");
            }

            return (await getConversations([uid], this.mailbox))[0];
        } catch (error) {
            logger.error(`[ConversationLocalDBProxy] Failed to get converstation ${uid}, ignore PROXY`, error);
        }

        return this.next!();
    }
    async byFolder(folder: string, sorted?: SortDescriptor): Promise<Array<string>> {
        try {
            if (!(await isLocal(folder, "mail_record"))) {
                return this.next!();
            }

            if (!isSynchronized(folder, "mail_record")) {
                await nudgeSync(folder, "mail_record");
            }

            const db = (await ContainerDatasource.retrieve<MailboxItem, number>("mail_record")) as MailboxRecordsDB;

            const conversations = await db.getConversationByFolder(folder);
            const result = sortBy(conversations, sorted?.fields?.at(0)?.column || "date");
            const uids: string[] = [];
            result.forEach(({ conversationUid, unseen, flagged }) => {
                if (matchFilter(unseen, flagged, sorted?.filter)) {
                    uids.push(conversationUid);
                }
            });
            return sorted?.fields?.at(0)?.dir === "Desc" ? uids.reverse() : uids;
        } catch (error) {
            logger.error(
                `[ConversationLocalDBProxy] Failed to fetch conversations by folder ${folder}, ignore PROXY`,
                error
            );
        }
        return this.next!();
    }
}

function matchFilter(
    unseen: boolean | undefined,
    flagged: boolean | undefined,
    filter: ItemFlagFilter = { must: [], mustNot: [] }
) {
    const flags = [!unseen && ItemFlag.Seen, flagged && ItemFlag.Important].filter(Boolean) as ItemFlag[];
    return filter.must?.every(flag => flags.includes(flag)) && !filter.mustNot?.some(flag => flags.includes(flag));
}

async function getConversations(uids: string[], mailbox: string) {
    const mailDB = (await ContainerDatasource.retrieve<MailboxItem, number>("mail_record")) as MailboxRecordsDB;
    const folderDB = await ContainerDatasource.retrieve<MailboxFolder, number>("mailboxacl");
    const folders = (await folderDB.getAllItems(mailbox)).reduce(
        (set, folder) => set.add(folder.uid!),
        new Set<string>()
    );
    return mailDB.getConversations(uids, folders);
}
