import { ItemFlag, ItemFlagFilter } from "@bluemind/core.container.api";
import { EventBusIdentifier } from "@bluemind/service-worker-bus/src/bus";
import ContainerDatasource from "@bluemind/service-worker-datasource";

import { synchronize } from "./Synchronize";
import SynchronizeQueue from "./SynchronizeQueue";

export const key: EventBusIdentifier<"SYNCHRONIZE", { uid: string; type: string; priority: number }> =
    "synchronization";

async function getContainerSyncStatus(uid: string, type: string) {
    const db = await ContainerDatasource.get(type);
    return db ? await db.getSyncStatus(uid) : undefined;
}

export async function isLocal(uid: string, type: string, filter?: ItemFlagFilter): Promise<boolean> {
    if (filter && !isFilterSynced(filter)) {
        return false;
    }
    return (await getContainerSyncStatus(uid, type)) !== undefined;
}

export function isSynchronized(uid: string, type: string): boolean {
    return !SynchronizeQueue.isInQueue({ uid, type });
}

export async function nudgeSync(uid: string, type: string): Promise<void> {
    const syncStatus = await getContainerSyncStatus(uid, type);

    if (!syncStatus) {
        throw new Error(`Container ${type}:${uid} is not subscribed`);
    }

    await SynchronizeQueue.flushOne({ uid, type });
}

export async function relaxedSynchronize(uid: string, type: string) {
    const db = await ContainerDatasource.get(type);
    if (db) {
        const syncStatus = await db.getSyncStatus(uid);
        if (syncStatus) {
            synchronize({ uid, type });
        }
    }
}

function isFilterSynced(filter: ItemFlagFilter) {
    const SYNC_FILTER: ItemFlagFilter = { must: [], mustNot: [ItemFlag.Deleted] };
    // All flag excluded from sync are also excluded for request
    // And all flag included for request are also included for sync.
    return (
        SYNC_FILTER.mustNot?.every(flag => filter.mustNot?.includes(flag)) &&
        SYNC_FILTER.must?.every(flag => filter.must?.includes(flag))
    );
}
