import { RouteHandler, RouteMatchCallback } from "workbox-core";

import { CacheFirst, NetworkOnly, StaleWhileRevalidate } from "workbox-strategies";

import { extensions } from "@bluemind/extensions";

import ResourceListProvider from "../../commons/ResourceListProvider";

import { WebAppIndexStrategy } from "./WebAppIndexStrategy";

type RouteConfig = {
    route: { capture: RouteMatchCallback; handler: RouteHandler };
    priority: number;
};
const PRIORITY = {
    MAX: 1024,
    DEFAULT: 128,
    MIN: 0
} as const;

ResourceListProvider.load();
export const defaultRoutes = {
    setup() {
        const routes: Record<string, RouteConfig> = {
            dev: {
                route: {
                    capture: ({ url, sameOrigin }) => url.hostname === "localhost" && !sameOrigin,
                    handler: new NetworkOnly()
                },
                priority: PRIORITY.MAX + 1
            },
            blank: {
                route: {
                    capture: ({ url }) => url.pathname === "/webapp/blank",
                    handler: new CacheFirst({ cacheName: "assets" })
                },
                priority: PRIORITY.DEFAULT
            },
            avatar: {
                route: {
                    capture: ({ sameOrigin, request, url }) =>
                        sameOrigin && request.destination === "image" && url.pathname.startsWith("/api/addressbooks"),
                    handler: new StaleWhileRevalidate({ cacheName: "avatar" })
                },
                priority: PRIORITY.DEFAULT
            },
            customStyle: {
                route: {
                    capture: ({ url }) => url.pathname.endsWith("style/customstyle.css"),
                    handler: new CacheFirst({ cacheName: "assets" })
                },
                priority: PRIORITY.MIN
            },
            peerDependencies: {
                route: {
                    capture: ({ url }) =>
                        ["/webapp/js/vue.js", "/webapp/js/vue-router.js", "/webapp/js/vuex.js"].includes(url.pathname),
                    handler: new CacheFirst({ cacheName: "assets" })
                },
                priority: PRIORITY.DEFAULT
            },
            index: {
                route: {
                    capture: ({ url }) => url.pathname === "/webapp/index.html",
                    handler: new WebAppIndexStrategy({ cacheName: "assets" })
                },
                priority: PRIORITY.MIN
            },
            handlers: {
                route: { capture: ResourceListProvider.isHandler, handler: new NetworkOnly() },
                priority: PRIORITY.MIN
            },
            assets: {
                route: {
                    capture: ResourceListProvider.isAsset,
                    handler: new CacheFirst({ cacheName: "assets" })
                },
                priority: PRIORITY.MIN
            },
            fallback: {
                route: {
                    capture: ({ request }) => request.destination === "document",
                    handler: new WebAppIndexStrategy({ cacheName: "assets" })
                },
                priority: PRIORITY.MIN
            }
        };

        Object.values(routes).forEach(route => {
            extensions.register("serviceworker.handlers", "net.bluemind.webapp.mail.js", {
                "route-handler": route
            });
        });
    }
};
