import { MailboxFoldersByOwnerClient } from "@bluemind/backend.mail.api";
import type { MailboxFolder, MailboxItem } from "@bluemind/backend.mail.api";
import type { ChangeSet, SyncService } from "@bluemind/commons.light/model/synchronization";
import type { ContainerChangeset } from "@bluemind/core.container.api";
import bus from "@bluemind/service-worker-bus/src/bus";
import ContainerDatasource, { type ContainerDB } from "@bluemind/service-worker-datasource";

import { key } from "../../Synchronization/SyncUtils";

export default class MailboxSyncService implements SyncService<MailboxFolder, number> {
    client: MailboxFoldersByOwnerClient;
    db: Promise<ContainerDB<MailboxFolder, number>>;

    constructor(uid: string, sid: string, domain: string) {
        const owner = uid.replace("mailbox:acls-", "");
        this.client = new MailboxFoldersByOwnerClient(sid, domain, owner);
        this.db = ContainerDatasource.retrieve<MailboxFolder, number>("mailboxacl");
    }
    async getRemoteChangeSet(since: number) {
        const { created, updated, deleted, version } = (await this.client.changesetById(since)) as Required<
            ContainerChangeset<number>
        >;
        const updatedIds = created.concat(updated);
        return { updated: updatedIds, deleted, version };
    }
    async getRemoteItems(ids: number[]) {
        const remoteItems = [];
        for (const id of ids) {
            const remoteItem = await this.client.getCompleteById(id);
            if (remoteItem) {
                bus.channel(key).emit("SYNCHRONIZE", { uid: remoteItem.uid!, type: "mail_record", priority: 0 });
                remoteItems.push(remoteItem);
            }
        }
        return remoteItems;
    }
    async getLocalChangeSet(containerUid: string) {
        const changeLog = await (await this.db).getChangeSet(containerUid);
        const { updated, deleted } = changeLog.reduce<{
            created: number[];
            updated: number[];
            deleted: number[];
        }>(
            (result, item) => {
                const { action, uid } = item;
                switch (action) {
                    case 0:
                        result.created?.push(uid);
                        break;
                    case 1:
                        result.updated?.push(uid);
                        break;
                    case 2:
                        result.deleted?.push(uid);
                        break;
                }

                return result;
            },
            { created: [], updated: [], deleted: [] }
        );
        return { updated, deleted, version: 0 };
    }
    async updateRemote(changeSet: ChangeSet<number>, containerUid: string) {
        const db = await this.db;
        await Promise.all(
            changeSet.updated.map(async id => {
                const itemValue = (await db.getItems([id], containerUid))[0];
                return this.client.updateById(id, itemValue.value);
            })
        );
        await Promise.all(
            changeSet.deleted.map(id => {
                return this.client.deleteById(id);
            })
        );
        await db.commit(changeSet.updated.concat(changeSet.deleted), containerUid);
    }

    async resetData(containerUid: string) {
        const db = await this.db;
        const items = await db.getAllItems(containerUid);

        const mailRecordDb = await ContainerDatasource.retrieve<MailboxItem, number>("mail_record");
        await Promise.all([
            db.reset(containerUid),
            items.map(({ uid }) => {
                mailRecordDb.reset(uid as string);
            })
        ]);
    }
}
