import { Conversation } from "@bluemind/backend.mail.api";
import { ItemFlag, ItemFlagFilter, SortDescriptor } from "@bluemind/core.container.api";

import { isLocal, isSynchronized, nudgeSync } from "../../Synchronization/SyncUtils";
import { setupMockDatasource, setupDatasourceError } from "../../testUtils/MockDatasource";

import ConversationLocalDBProxy from "./ConversationLocalDBProxy";
import { ConversationStub } from "./MailboxRecordsDB";

const mockNext = vi.fn().mockResolvedValue(undefined);

vi.mock("@bluemind/service-worker-datasource", () => ({
    default: {
        retrieve: vi.fn()
    }
}));

vi.mock("@bluemind/logger", () => ({
    default: {
        error: vi.fn()
    }
}));

vi.mock("../../Synchronization/SyncUtils", () => ({
    isLocal: vi.fn(),
    isSynchronized: vi.fn(),
    nudgeSync: vi.fn()
}));

const setupProxy = () => new ConversationLocalDBProxy("api-key", "mailbox!uid");

describe("ConversationLocalDBProxy", () => {
    let mockDb: any;
    beforeEach(() => {
        vi.mocked(isLocal).mockResolvedValue(true);
        vi.mocked(isSynchronized).mockReturnValue(true);
        vi.mocked(nudgeSync).mockResolvedValue();

        const datasource = setupMockDatasource({
            getConversations: vi
                .fn()
                .mockResolvedValue([{ conversationUid: "1" }, { conversationUid: "2" }] as Conversation[]),
            getConversationByFolder: vi.fn().mockResolvedValue([
                {
                    conversationUid: "1",
                    date: 1,
                    size: 1,
                    subject: "subject 1",
                    sender: "sender 1",
                    unseen: true,
                    flagged: false
                },
                {
                    conversationUid: "2",
                    date: 2,
                    size: 2,
                    subject: "subject 2",
                    sender: "sender 2",
                    unseen: true,
                    flagged: false
                },
                {
                    conversationUid: "3",
                    date: 3,
                    size: 3,
                    subject: "subject 3",
                    sender: "sender 3",
                    unseen: false,
                    flagged: true
                },
                {
                    conversationUid: "4",
                    date: 4,
                    size: 4,
                    subject: "subject 4",
                    sender: "sender 4",
                    unseen: false,
                    flagged: true
                }
            ] as ConversationStub[])
        });
        mockDb = datasource.mockDb;
    });

    afterEach(() => {
        vi.clearAllMocks();
    });

    describe("multipleGet()", () => {
        test("retrieves items", async () => {
            const proxy = setupProxy();

            const result = await proxy.multipleGet(["1", "2"]);

            expect(mockDb.getConversations).toHaveBeenCalledOnce();
            expect(result).toHaveLength(2);
        });

        test("fallbacks to next when datasource fails", async () => {
            setupDatasourceError();
            const proxy = setupProxy();
            proxy.next = mockNext;

            const result = await proxy.multipleGet(["1", "2"]);

            expect(proxy.next).toHaveBeenCalledOnce();
            expect(result).toBeUndefined();
        });

        test("should got next if its not local", async () => {
            vi.mocked(isLocal).mockResolvedValue(false);

            const proxy = setupProxy();
            proxy.next = mockNext;

            const result = await proxy.multipleGet(["1", "2"]);

            expect(proxy.next).toHaveBeenCalledOnce();
            expect(result).toBeUndefined();
        });

        test("should nudge if it's needed", async () => {
            vi.mocked(isSynchronized).mockReturnValue(false);

            const proxy = setupProxy();
            proxy.next = mockNext;

            const result = await proxy.multipleGet(["1", "2"]);

            expect(nudgeSync).toHaveBeenCalledOnce();
            expect(result).toBeDefined();
        });
    });

    describe("get()", () => {
        test("retrieves items", async () => {
            const proxy = setupProxy();

            const result = await proxy.get("1");

            expect(mockDb.getConversations).toHaveBeenCalledOnce();
            expect(result.conversationUid).toBe("1");
        });

        test("fallbacks to next when datasource fails", async () => {
            setupDatasourceError();
            const proxy = setupProxy();
            proxy.next = mockNext;

            const result = await proxy.get("1");

            expect(proxy.next).toHaveBeenCalledOnce();
            expect(result).toBeUndefined();
        });

        test("should got next if its not local", async () => {
            vi.mocked(isLocal).mockResolvedValue(false);

            const proxy = setupProxy();
            proxy.next = mockNext;

            const result = await proxy.get("1");

            expect(proxy.next).toHaveBeenCalledOnce();
            expect(result).toBeUndefined();
        });

        test("should nudge if it's needed", async () => {
            vi.mocked(isSynchronized).mockReturnValue(false);

            const proxy = setupProxy();
            proxy.next = mockNext;

            const result = await proxy.get("1");

            expect(nudgeSync).toHaveBeenCalledOnce();
            expect(result).toBeDefined();
        });
    });

    describe("byFolder()", () => {
        test("retrieves items without filter", async () => {
            const proxy = setupProxy();

            const result = await proxy.byFolder("folder");

            expect(mockDb.getConversationByFolder).toHaveBeenCalledOnce();
            expect(result).toHaveLength(4);
        });

        test("fallbacks to next when datasource fails", async () => {
            setupDatasourceError();
            const proxy = setupProxy();
            proxy.next = mockNext;

            const result = await proxy.get("1");

            expect(proxy.next).toHaveBeenCalledOnce();
            expect(result).toBeUndefined();
        });

        test("retrieves items with mustnot flag filter ", async () => {
            const sortDescriptor: {
                fields: SortDescriptor.Field[];
                filter: ItemFlagFilter;
            } = {
                fields: [
                    {
                        column: "date",
                        dir: "Desc"
                    }
                ],
                filter: { must: [], mustNot: [ItemFlag.Seen] }
            };
            const proxy = setupProxy();

            const result = await proxy.byFolder("folder", sortDescriptor);

            expect(mockDb.getConversationByFolder).toHaveBeenCalledOnce();
            expect(result[1]).toBe("1");
            expect(result).toHaveLength(2);
        });

        test("retrieves items with must flag filter ", async () => {
            const sortDescriptor: {
                fields: SortDescriptor.Field[];
                filter: ItemFlagFilter;
            } = {
                fields: [
                    {
                        column: "date",
                        dir: "Asc"
                    }
                ],
                filter: { must: [ItemFlag.Important], mustNot: [] }
            };
            const proxy = setupProxy();

            const result = await proxy.byFolder("folder", sortDescriptor);

            expect(mockDb.getConversationByFolder).toHaveBeenCalledOnce();
            expect(result[1]).toBe("4");
            expect(result).toHaveLength(2);
        });

        test("should got next if its not local", async () => {
            vi.mocked(isLocal).mockResolvedValue(false);

            const proxy = setupProxy();
            proxy.next = mockNext;

            const result = await proxy.byFolder("folder");

            expect(proxy.next).toHaveBeenCalledOnce();
            expect(result).toBeUndefined();
        });

        test("should nudge synchro if needed", async () => {
            vi.mocked(isSynchronized).mockReturnValue(false);

            const proxy = setupProxy();
            proxy.next = mockNext;

            const result = await proxy.byFolder("folder");

            expect(nudgeSync).toHaveBeenCalledOnce();
            expect(result).toBeDefined();
        });
    });
});
