package net.bluemind.webmodules.devmode.configurator;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonInclude.Include;

import io.vertx.core.Vertx;
import io.vertx.core.http.HttpMethod;
import io.vertx.core.http.HttpServerRequest;
import io.vertx.core.http.HttpServerResponse;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.webmodule.project.vite.DevConfiguration;
import net.bluemind.webmodule.server.SecurityConfig;
import net.bluemind.webmodule.server.WebModule;

public abstract class DevConfigurationHandler {

	@JsonInclude(Include.NON_NULL)
	protected static class Status {
		private final String status;
		private final String message;
		private final DevConfiguration value;

		public Status(String status) {
			this(status, null);
		}

		public Status(String status, String message) {
			this.status = status;
			this.message = message;
			this.value = DevConfiguration.get();
		}

		public String getStatus() {
			return status;
		}

		public boolean getCSP() {
			return SecurityConfig.cspHeader;
		}

		public boolean getDevMode() {
			return DevConfiguration.DEV_MODE;
		}

		public String getMessage() {
			return message;
		}

		public DevConfiguration getValue() {
			return value;
		}

		public static Status ERROR(String message) {
			return new Status("error", message);
		}

		public static Status OK(String message) {
			return new Status("ok", message);
		}
	}

	public static DevConfigurationHandler create(HttpServerRequest request, WebModule module, Vertx vertx) {
		String bundle = request.getParam("bundle");
		if (bundle != null) {
			return new BundeConfigurationHandler(bundle, module, vertx);
		}
		return new GlobalConfigurationHandler(module, vertx);

	}

	protected Vertx vertx;
	protected WebModule module;

	public DevConfigurationHandler(WebModule module, Vertx vertx) {
		this.module = module;
		this.vertx = vertx;
	}

	public void execute(HttpServerRequest request) {
		HttpMethod method = request.method();
		if (method.equals(HttpMethod.POST)) {
			enable(request);
		} else if (method.equals(HttpMethod.PUT)) {
			update(request);
		} else if (method.equals(HttpMethod.DELETE)) {
			disable(request);
		} else if (method.equals(HttpMethod.GET)) {
			get(request);
		} else {
			sendNotAllowed(request, "Methode %s is not allowed", method);
		}
	}

	protected abstract void update(HttpServerRequest request);

	protected abstract void get(HttpServerRequest request);

	protected abstract void disable(HttpServerRequest request);

	protected abstract void enable(HttpServerRequest request);

	protected void sendOk(HttpServerRequest request, String message, Object... args) {
		if (message != null) {
			message = String.format(message, args);
		}
		HttpServerResponse resp = request.response();
		resp.putHeader("Content-Type", "application/json");
		resp.setStatusCode(200);
		resp.end(JsonUtils.asString(Status.OK(message)));
	}

	protected void sendNotFound(HttpServerRequest request, String message, Object... args) {
		if (message != null) {
			message = String.format(message, args);
		}
		HttpServerResponse resp = request.response();
		resp.putHeader("Content-Type", "application/json");
		resp.setStatusCode(404);
		resp.end(JsonUtils.asString(Status.ERROR(message)));

	}

	protected void sendNotAllowed(HttpServerRequest request, String message, Object... args) {
		if (message != null) {
			message = String.format(message, args);
		}
		HttpServerResponse resp = request.response();
		resp.putHeader("Content-Type", "application/json");
		resp.setStatusCode(405);
		resp.end(JsonUtils.asString(Status.ERROR(message)));

	}
}
