/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.webmodule.authenticationfilter.internal;

import java.util.Base64;
import java.util.Base64.Decoder;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.http.HttpServerRequest;
import io.vertx.core.json.DecodeException;
import io.vertx.core.json.JsonObject;

public class StateParameter {
	private static final Logger logger = LoggerFactory.getLogger(StateParameter.class);

	private static final Decoder b64UrlDecoder = Base64.getUrlDecoder();

	private final String path;
	public final String codeVerifierKey;
	public final String domainUid;

	private StateParameter(String path, String codeVerifierKey, String domainUid) {
		this.path = path;
		this.codeVerifierKey = codeVerifierKey;
		this.domainUid = domainUid;
	}

	public static StateParameter fromRequest(HttpServerRequest request, List<String> forwadedFor) {
		String stateAsString = request.params().get("state");
		if (stateAsString == null || stateAsString.isBlank()) {
			if (logger.isDebugEnabled()) {
				logger.error("[{}][{}] 'state' parameter is null or blank", forwadedFor, request.path());
			}
			return null;
		}

		try {
			JsonObject jsonStateParameter = new JsonObject(new String(b64UrlDecoder.decode(stateAsString.getBytes())));

			return new StateParameter(jsonStateParameter.getString("path"),
					jsonStateParameter.getString("codeVerifierKey"), jsonStateParameter.getString("domain_uid"));
		} catch (DecodeException | IllegalArgumentException | IndexOutOfBoundsException e) {
			if (logger.isDebugEnabled()) {
				logger.error("[{}][{}] invalid 'state' parameter", forwadedFor, request.path(), e);
			}

			return null;
		}
	}

	public String getRedirectTo() {
		if (path == null) {
			return "/";
		}

		return path;
	}
}
