/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.webmodule.authenticationfilter;

import java.nio.charset.StandardCharsets;
import java.util.Base64;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.AbstractVerticle;
import io.vertx.core.AsyncResult;
import io.vertx.core.MultiMap;
import io.vertx.core.Verticle;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.http.HttpClient;
import io.vertx.core.http.HttpClientOptions;
import io.vertx.core.http.HttpClientRequest;
import io.vertx.core.http.HttpClientResponse;
import io.vertx.core.http.HttpHeaders;
import io.vertx.core.http.HttpMethod;
import io.vertx.core.http.RequestOptions;
import io.vertx.core.json.JsonObject;
import net.bluemind.keycloak.api.IKeycloakUids;
import net.bluemind.keycloak.utils.endpoint.KeycloakEndpoints;
import net.bluemind.keydb.sessiondata.SessionData;
import net.bluemind.keydb.sessiondata.SessionDataStore;
import net.bluemind.lib.vertx.IUniqueVerticleFactory;
import net.bluemind.lib.vertx.IVerticleFactory;

public class OpenIdRefreshHandler extends AbstractVerticle {
	private static final Logger logger = LoggerFactory.getLogger(OpenIdRefreshHandler.class);

	public static class OpenIdRefreshHandlerVerticleFactory implements IVerticleFactory, IUniqueVerticleFactory {
		@Override
		public boolean isWorker() {
			return true;
		}

		@Override
		public Verticle newInstance() {
			return new OpenIdRefreshHandler();
		}
	}

	private class SessionRefresher {
		private final SessionData sessionData;

		public SessionRefresher(SessionData sessionData) {
			this.sessionData = sessionData;
		}

		private void refresh() {
			if (logger.isDebugEnabled()) {
				logger.debug("Refresh session {}, using endpoint: {}", sessionData.authKey,
						KeycloakEndpoints.tokenEndpoint(sessionData.realm));
			}

			httpClient.request(new RequestOptions().setMethod(HttpMethod.POST)
					.setAbsoluteURI(KeycloakEndpoints.tokenEndpoint(sessionData.realm)), this::decorateRequest);
		}

		private void decorateRequest(AsyncResult<HttpClientRequest> request) {
			if (!request.succeeded()) {
				logger.error("Unable to refresh session {}", sessionData.authKey, request.cause());
				return;
			}

			HttpClientRequest r = request.result();
			r.response(this::refreshResponse);

			if (logger.isDebugEnabled()) {
				String[] chunks = sessionData.jwtToken.getString("refresh_token").split("\\.");
				logger.debug("Refresh token: chunk0: {}, chunk1: {}",
						new String(Base64.getUrlDecoder().decode(chunks[0])),
						new String(Base64.getUrlDecoder().decode(chunks[1])));
			}

			byte[] postData = new StringBuilder().append("client_id=").append(IKeycloakUids.clientId(sessionData.realm)) //
					.append("&client_secret=").append(sessionData.openIdClientSecret)
					.append("&grant_type=refresh_token&refresh_token=")
					.append(sessionData.jwtToken.getString("refresh_token")).toString()
					.getBytes(StandardCharsets.UTF_8);

			MultiMap headers = r.headers();
			headers.add(HttpHeaders.ACCEPT_CHARSET, StandardCharsets.UTF_8.name());
			headers.add(HttpHeaders.CONTENT_TYPE, "application/x-www-form-urlencoded");
			headers.add(HttpHeaders.CONTENT_LENGTH, Integer.toString(postData.length));
			r.write(Buffer.buffer(postData));
			r.end();
		}

		private void refreshResponse(AsyncResult<HttpClientResponse> response) {
			if (!response.succeeded()) {
				logger.error("Fail to refresh session {}", sessionData.authKey, response.cause());
				return;
			}

			response.result().body(body -> {
				if (response.result().statusCode() != 200) {
					logger.error("Fail to refresh session {}, response status {}, content {}", sessionData.authKey,
							response.result().statusCode(), new String(body.result().getBytes()));
					return;
				}

				JsonObject jwtToken = new JsonObject(new String(body.result().getBytes()));

				SessionDataStore.get()
						.updateSessionData(sessionData.setOpenId(jwtToken, sessionData.realm,
								sessionData.openIdClientSecret, sessionData.internalAuth,
								System.currentTimeMillis() + SessionDataStore.SESSIONID_REFRESH_PERIOD));

				if (logger.isDebugEnabled()) {
					logger.debug("Session {} refreshed successfully", sessionData.authKey);
				}
			});
		}
	}

	private HttpClient httpClient;

	private OpenIdRefreshHandler() {
	}

	@Override
	public void start() throws Exception {
		httpClient = vertx.createHttpClient(new HttpClientOptions().setTrustAll(true).setVerifyHost(false));

		vertx.setPeriodic(60_000, 60_000, tid -> SessionDataStore.get().requeueRefreshSessionId());
		vertx.setPeriodic(90_000, 60_000, tid -> refresh());
	}

	public void refresh() {
		String sessionId;
		while ((sessionId = SessionDataStore.get().getSessionIdToRefresh()) != null) {
			SessionData sessionData = SessionDataStore.get().getIfPresent(sessionId);

			if (sessionData == null) {
				if (logger.isDebugEnabled()) {
					logger.debug("Remove unknown session {} from refresh queue - session logged out ?", sessionId);
				}
				SessionDataStore.get().removeRefreshSessionId(sessionId);
				continue;
			}

			if (sessionData.openIdRefreshStamp > System.currentTimeMillis()) {
				// No refresh needed
				return;
			}

			new SessionRefresher(sessionData).refresh();
		}
	}
}
