/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2023
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.webmodule.authenticationfilter;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Strings;

import io.vertx.core.Vertx;
import net.bluemind.authentication.api.IAuthenticationPromise;
import net.bluemind.authentication.api.LoginResponse;
import net.bluemind.authentication.api.LoginResponse.Status;
import net.bluemind.config.Token;
import net.bluemind.core.api.AsyncHandler;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.http.HttpClientProvider;
import net.bluemind.core.rest.http.ILocator;
import net.bluemind.core.rest.http.VertxPromiseServiceProvider;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.IDomainsPromise;
import net.bluemind.keydb.sessiondata.SessionData;
import net.bluemind.keydb.sessiondata.SessionDataStore;
import net.bluemind.mailbox.api.IMailboxesPromise;
import net.bluemind.mailbox.api.Mailbox;
import net.bluemind.mailbox.api.Mailbox.Type;
import net.bluemind.network.topology.Topology;
import net.bluemind.server.api.TagDescriptor;
import net.bluemind.webmodule.authenticationfilter.internal.ExternalCreds;

public class AuthProvider {
	private static final Logger logger = LoggerFactory.getLogger(AuthProvider.class);

	private record CredentialInfosWithDomain(ExternalCreds externalCreds, String userDomainUid,
			ItemValue<Domain> domain) {
	}

	private record CredentialInfosWithMailbox(ExternalCreds externalCreds, String userDomainUid,
			ItemValue<Mailbox> mailbox) {
	}

	public static final int DEFAULT_MAX_SESSIONS_PER_USER = 5;
	private static final String BM_WEBSERVER_AUTHFILTER = "bm-webserver-authfilter";

	private HttpClientProvider clientProvider;

	private final String domainUid;
	private final boolean internalAuth;

	public AuthProvider(Vertx vertx, String domainUid) {
		clientProvider = new HttpClientProvider(vertx);
		this.domainUid = domainUid;
		this.internalAuth = false;
	}

	public AuthProvider(Vertx vertx, String domainUd, boolean internalAuth) {
		clientProvider = new HttpClientProvider(vertx);
		this.domainUid = domainUd;
		this.internalAuth = internalAuth;
	}

	public void sessionId(ExternalCreds externalCreds, List<String> remoteIps, AsyncHandler<SessionData> handler) {
		if (Strings.isNullOrEmpty(externalCreds.getLoginAtDomain())
				|| !externalCreds.getLoginAtDomain().contains("@")) {
			handler.failure(new ServerFault(
					"Invalid loginAtDomain " + externalCreds.getLoginAtDomain() + " from external credentials",
					ErrorCode.FORBIDDEN));
			return;
		}

		if ("admin0@global.virt".equals(externalCreds.getLoginAtDomain())) {
			if (internalAuth) {
				// No admin0@global.virt authentication using external authentication service
				doSudo(remoteIps, externalCreds, handler);
			} else {
				handler.failure(new ServerFault(
						"Authentication failure: admin0@global.virt only available from internal authentication",
						ErrorCode.FORBIDDEN));
			}

			return;
		}

		externalCredsToMailbox(remoteIps, externalCreds, handler);
	}

	public void sessionId(final String loginAtDomain, final String password, List<String> remoteIps,
			final AsyncHandler<SessionData> handler) {
		VertxPromiseServiceProvider sp = getProvider(null, remoteIps);

		logger.info("authenticating {}", loginAtDomain);
		IAuthenticationPromise auth = sp.instance(TagDescriptor.bm_core.getTag(), IAuthenticationPromise.class);
		auth.loginWithParams(loginAtDomain.toLowerCase(), password, BM_WEBSERVER_AUTHFILTER, true).exceptionally(e -> {
			logger.error("error during authentication of {}", loginAtDomain, e);
			handler.failure(new ServerFault("error login: No server assigned or server not avalaible"));
			return null;
		}).thenAccept(lr -> {
			logger.info("Authenticated {}, response: {}", loginAtDomain, lr.status);
			if (lr.status == Status.Ok || lr.status == Status.Expired) {
				handlerLoginSuccess(lr, handler);
			} else {
				handler.failure(new ServerFault("error during login " + lr.message, ErrorCode.INVALID_PASSWORD));
			}
		});
	}

	private void doSudo(List<String> remoteIps, ExternalCreds externalCreds, final AsyncHandler<SessionData> handler) {
		getProvider(Token.admin0(), remoteIps).instance(IAuthenticationPromise.class)
				.suWithParams(externalCreds.getLoginAtDomain(), true).exceptionally(t -> null).thenAccept(lr -> {
					if (lr == null) {
						handler.failure(
								new ServerFault("Error during sudo for user " + externalCreds.getLoginAtDomain()));
						return;
					}

					if (lr.status == Status.Ok) {
						handlerLoginSuccess(lr, handler);
					} else {
						handler.success(null);
					}
				});
	}

	private void externalCredsToMailbox(List<String> remoteIps, ExternalCreds externalCreds,
			AsyncHandler<SessionData> handler) {
		// If BlueMind internal authentication and global.virt realm then trust
		// ExternalCreds domain part otherwise use domainUid from OpenID state parameter
		String userDomainUid = (internalAuth && domainUid.equals("global.virt"))
				? externalCreds.getLoginAtDomain().split("@")[1]
				: domainUid;

		VertxPromiseServiceProvider provider = getProvider(Token.admin0(), remoteIps);
		CompletableFuture<ItemValue<Mailbox>> mailboxFuture;
		if (externalCreds.getUserUid() != null && !externalCreds.getUserUid().isBlank()) {
			mailboxFuture = getMailboxByUid(provider, externalCreds, userDomainUid);
		} else {
			mailboxFuture = getMailboxByEmail(provider, externalCreds, userDomainUid);
		}

		// BM-21397: don't search byName, only byEmail
		mailboxFuture.whenComplete((mailbox, mailboxException) -> {
			if (mailboxException != null) {
				handler.failure(mailboxException);
				return;
			}

			if (mailbox != null) {
				sudoFromMailbox(remoteIps, new CredentialInfosWithMailbox(externalCreds, userDomainUid, mailbox),
						handler);
			} else if (internalAuth) {
				// Mailbox does not exists, try on-the-fly import
				logger.info("Try sudo with login {} (try on-the-fly import)", externalCreds.getLoginAtDomain());
				doSudo(remoteIps, externalCreds, handler);
			} else {
				// Mailbox does not exists but external authentication server, try on-the-fly
				// import
				provider.instance(IDomainsPromise.class)
						.findByNameOrAliases(externalCreds.getLoginAtDomain().split("@")[1])
						.whenComplete((domain, domainException) -> {
							if (domainException != null) {
								handler.failure(domainException);
								return;
							}

							onTheFlyImportFromExternalAuth(remoteIps,
									new CredentialInfosWithDomain(externalCreds, userDomainUid, domain), handler);
						});
			}
		});
	}

	private CompletableFuture<ItemValue<Mailbox>> getMailboxByEmail(VertxPromiseServiceProvider provider,
			ExternalCreds externalCreds, String userDomainUid) {
		logger.info("Searching mailbox with email: {}", externalCreds.getLoginAtDomain());
		return provider.instance(IMailboxesPromise.class, userDomainUid).byEmail(externalCreds.getLoginAtDomain());
	}

	private CompletableFuture<ItemValue<Mailbox>> getMailboxByUid(VertxPromiseServiceProvider provider,
			ExternalCreds externalCreds, String userDomainUid) {
		logger.info("Getting mailbox uid: {}", externalCreds.getUserUid());
		return provider.instance(IMailboxesPromise.class, userDomainUid).getComplete(externalCreds.getUserUid());
	}

	private void onTheFlyImportFromExternalAuth(List<String> remoteIps,
			CredentialInfosWithDomain credentialInfosWithDomain, AsyncHandler<SessionData> handler) {
		// BM-21416: Ensure external credentials domain match BlueMind domain before
		// trying on-the-fly import
		if (!credentialInfosWithDomain.domain.uid.equals(credentialInfosWithDomain.userDomainUid)) {
			handler.failure(new ServerFault("Authentication failure: external credentials "
					+ credentialInfosWithDomain.externalCreds.getLoginAtDomain() + " not from domain: " + domainUid,
					ErrorCode.FORBIDDEN));
			return;
		}

		logger.info("Try sudo with login {} (try on-the-fly import)",
				credentialInfosWithDomain.externalCreds.getLoginAtDomain());
		doSudo(remoteIps, credentialInfosWithDomain.externalCreds, handler);
	}

	private void sudoFromMailbox(List<String> remoteIps, CredentialInfosWithMailbox credentialInfosWithMailbox,
			AsyncHandler<SessionData> handler) {
		if (credentialInfosWithMailbox.mailbox.value.type != Type.user
				|| credentialInfosWithMailbox.mailbox.value.archived) {
			handler.success(null);
			return;
		}

		String realLoginAtdomain = credentialInfosWithMailbox.mailbox.value.name + "@"
				+ credentialInfosWithMailbox.userDomainUid;

		logger.info("Try sudo with login {} (Submitted login {})", realLoginAtdomain,
				credentialInfosWithMailbox.externalCreds.getLoginAtDomain());
		credentialInfosWithMailbox.externalCreds.setLoginAtDomain(realLoginAtdomain);

		doSudo(remoteIps, credentialInfosWithMailbox.externalCreds, handler);
	}

	private void handlerLoginSuccess(LoginResponse lr, AsyncHandler<SessionData> handler) {
		try {
			SessionData sd = new SessionData(lr);
			SessionDataStore.get().put(sd);

			handler.success(sd);
		} catch (Exception e) {
			handler.failure(e);
		}
	}

	public CompletableFuture<Void> logout(String sid) {
		SessionData sessionData = SessionDataStore.get().getIfPresent(sid);

		if (sessionData != null) {
			return logout(sessionData);
		}

		return logout("Unknown", sid);
	}

	public CompletableFuture<Void> logout(SessionData sessionData) {
		return logout(sessionData.loginAtDomain, sessionData.authKey);
	}

	/**
	 * Logout on core as core HZ message will invalidate webserver session
	 * 
	 * See {@link AuthenticationFilter#setVertx(Vertx)}
	 * 
	 * @param id
	 * @param sessionId
	 * @return
	 */
	private CompletableFuture<Void> logout(String id, String sessionId) {
		logger.info("Log out session {} for {}", sessionId, id);
		return getProvider(sessionId, Collections.emptyList()).instance(IAuthenticationPromise.class).logout()
				.whenComplete((v, fn) -> {
					if (fn != null) {
						logger.warn(fn.getMessage());
					}
				});
	}

	private VertxPromiseServiceProvider getProvider(String apiKey, List<String> remoteIps) {
		ILocator lc = (String service, AsyncHandler<String[]> asyncHandler) -> asyncHandler.success(
				new String[] { Topology.get().anyIfPresent(service).map(s -> s.value.address()).orElse("127.0.0.1") });
		VertxPromiseServiceProvider prov = new VertxPromiseServiceProvider(clientProvider, lc, apiKey, remoteIps);
		prov.setOrigin(BM_WEBSERVER_AUTHFILTER);
		return prov;
	}
}
