/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2025
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.user.service.internal;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import net.bluemind.addressbook.api.IAddressBookUids;
import net.bluemind.calendar.api.ICalendarUids;
import net.bluemind.core.container.api.ContainerQuery;
import net.bluemind.core.container.api.IContainerManagement;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.model.BaseContainerDescriptor;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.acl.AccessControlEntry;
import net.bluemind.core.container.model.acl.Verb;
import net.bluemind.core.container.repair.ContainerRepairOp;
import net.bluemind.core.container.service.internal.Containers;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.service.RepairTaskMonitor;
import net.bluemind.mailbox.api.IMailboxAclUids;
import net.bluemind.notes.api.INoteUids;
import net.bluemind.todolist.api.ITodoUids;

/**
 * Here we want a repair to ensure ACL consistency on shared mailboxes.
 * Currently, a hook/sanitizer system allows rights given to a shared mailbox to
 * be propagated to other containers.
 * 
 * This repair allows this propagation to be (re)done if it is not consistent.
 */
public class SharedMailboxAclRepair implements ContainerRepairOp {

	@Override
	public void check(BmContext context, String domainUid, DirEntry entry, RepairTaskMonitor monitor) {
		String sharedMailboxUid = entry.entryUid;
		String containerUid = IMailboxAclUids.uidForMailbox(sharedMailboxUid);
		Runnable maintenance = () -> {
			monitor.log("ACL Check for shared mailbox container {}", sharedMailboxUid);
			getMissingAclForSharedMailboxContainers(monitor, context, containerUid, entry.entryUid, domainUid);
		};

		verifyContainer(domainUid, monitor, maintenance, containerUid);
	}

	private void logCheck(RepairTaskMonitor monitor, Map<ContainerDescriptor, List<AccessControlEntry>> checkMap) {
		for (Entry<ContainerDescriptor, List<AccessControlEntry>> checkEntry : checkMap.entrySet()) {
			if (checkEntry.getValue().isEmpty()) {
				String msg = """
						ACL compliant for container %s(%s)
						""".formatted(checkEntry.getKey().uid, checkEntry.getKey().type);
				monitor.log(msg);
			} else {
				String msg = """
						ACL NOT compliant for container %s(%s)
						Missing %s
						""".formatted(checkEntry.getKey().uid, checkEntry.getKey().type, checkEntry.getValue());
				monitor.warn(msg);
			}
		}
	}

	private void logRepair(RepairTaskMonitor monitor, ContainerDescriptor container, List<AccessControlEntry> acls) {
		if (acls.isEmpty()) {
			String msg = """
					No Repairing ACL for container %s(%s)
					""".formatted(container.uid, container.type);
			monitor.log(msg);
		} else {
			String msg = """
					Repairing ACL for container %s(%s)
					=> %s
					""".formatted(container.uid, container.type, acls);
			monitor.log(msg);
		}
	}

	@Override
	public void repair(BmContext context, String domainUid, DirEntry entry, RepairTaskMonitor monitor) {
		if (entry.kind != Kind.SHARED_MAILBOX) {
			return;
		}
		String sharedMailboxUid = entry.entryUid;
		String containerUid = IMailboxAclUids.uidForMailbox(sharedMailboxUid);
		Runnable maintenance = () -> {
			Map<ContainerDescriptor, List<AccessControlEntry>> checkMap = getMissingAclForSharedMailboxContainers(
					monitor, context, containerUid, entry.entryUid, domainUid);
			monitor.log("ACL Repairing for shared mailbox container {} with {} missing acl containers",
					sharedMailboxUid, String.valueOf(checkMap.size()));
			repairInconsistentAcl(monitor, checkMap);
		};

		verifyContainer(domainUid, monitor, maintenance, containerUid);
	}

	@Override
	public void verifyContainer(String domainUid, RepairTaskMonitor monitor, Runnable maintenance,
			String containerUid) {

		BaseContainerDescriptor container = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM)
				.instance(IContainers.class).getLightIfPresent(containerUid);

		if (container != null) {
			maintenance.run();
		}
	}

	private static List<AccessControlEntry> aclOnSharedMailbox(String containerUid) {
		ServerSideServiceProvider provider = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);
		IContainerManagement containerMgmtApi = provider.instance(IContainerManagement.class, containerUid);
		return containerMgmtApi.getAccessControlList().stream() //
				.filter(ac -> !ac.subject.equals(containerUid)).toList();
	}

	public static final List<String> containerTypes = Arrays.asList(ICalendarUids.TYPE, IAddressBookUids.TYPE,
			ITodoUids.TYPE, INoteUids.TYPE);

	private static List<ContainerDescriptor> getContainersByOwner(String owner) {
		ServerSideServiceProvider provider = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);
		IContainers containerService = new Containers(provider.getContext());
		return containerTypes.stream() //
				.map(containerType -> ContainerQuery.ownerAndType(owner, containerType))
				.flatMap(query -> containerService.all(query).stream()).toList();
	}

	private void repairInconsistentAcl(RepairTaskMonitor monitor,
			Map<ContainerDescriptor, List<AccessControlEntry>> checkMap) {

		List<Entry<ContainerDescriptor, List<AccessControlEntry>>> containerAclEntries = checkMap.entrySet().stream() //
				.filter(container -> !container.getValue().isEmpty()).toList();

		for (Entry<ContainerDescriptor, List<AccessControlEntry>> container : containerAclEntries) {
			IContainerManagement containerMgmtApi = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM)
					.instance(IContainerManagement.class, container.getKey().uid);
			List<AccessControlEntry> aclContainer = containerMgmtApi.getAccessControlList();
			aclContainer.addAll(container.getValue());
			logRepair(monitor, container.getKey(), aclContainer);
			containerMgmtApi.setAccessControlList(aclContainer);
		}
	}

	private Map<ContainerDescriptor, List<AccessControlEntry>> getMissingAclForSharedMailboxContainers(
			RepairTaskMonitor monitor, BmContext context, String containerUid, String owner, String domainUid) {

		List<AccessControlEntry> aclOnSharedMailbox = aclOnSharedMailbox(containerUid);
		List<ContainerDescriptor> sharedMailboxContainers = getContainersByOwner(owner);

		Map<ContainerDescriptor, List<AccessControlEntry>> missingVerbs = new HashMap<>();

		List<String> otherMailboxOwners = aclOnSharedMailbox.stream().map(ac -> ac.subject)
				.filter(s -> !s.equals(owner)).distinct().toList();
		for (String mailboxOwner : otherMailboxOwners) {
			List<AccessControlEntry> containerAclOnMailbox = AccessControlEntry
					.compact(aclOnSharedMailbox.stream().filter(ac -> ac.subject.equals(mailboxOwner)).toList());

			List<ContainerDescriptor> containersByOwner = getContainersByOwner(mailboxOwner);

			sharedMailboxContainers.forEach(sharedContainer -> {
				IContainerManagement containerMgmtApi = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM)
						.instance(IContainerManagement.class, sharedContainer.uid);
				List<AccessControlEntry> aclOnSharedContainer = AccessControlEntry
						.compact(containerMgmtApi.getAccessControlList().stream().toList());

				List<AccessControlEntry> containerHasAccessOnto = AccessControlEntry
						.compact(aclOnSharedContainer.stream().filter(ac -> ac.subject.equals(mailboxOwner)).toList());

				List<AccessControlEntry> containerAclOnMailboxCpy = new ArrayList<AccessControlEntry>();
				containerAclOnMailboxCpy.addAll(containerAclOnMailbox);
				if (!containerHasAccessOnto.isEmpty()) {
					containerAclOnMailboxCpy.removeIf(v -> containerHasAccessOnto.contains(v));
				}

				if (!containerAclOnMailboxCpy.isEmpty()) {
					containersByOwner.stream() //
							.filter(cd -> cd.type.equals(sharedContainer.type)) //
							.filter(container -> samePrefix(sharedContainer, container)).findFirst()
							.ifPresent(container -> {
								List<AccessControlEntry> aclToSet = new ArrayList<>();
								aclToSet.addAll(containerAclOnMailboxCpy.stream()
										.map(ac -> AccessControlEntry.create(container.owner, ac.verb)).toList());
								if (sharedContainer.type.equals(ICalendarUids.TYPE)
										&& aclToSet.stream().anyMatch(ac -> ac.verb.can(Verb.Write))) {
									aclToSet.add(AccessControlEntry.create(container.owner, Verb.ReadExtended));
								}
								missingVerbs.put(sharedContainer, aclToSet);
							});
				} else {
					missingVerbs.put(sharedContainer, Collections.emptyList());
				}
			});
		}

		logCheck(monitor, missingVerbs);
		return missingVerbs;
	}

	private boolean samePrefix(ContainerDescriptor sharedContainer, ContainerDescriptor container) {
		String prefixSharedContainer = sharedContainer.uid.replace(sharedContainer.owner, "");
		String prefixContainer = container.uid.replace(container.owner, "");
		return prefixContainer.equals(prefixSharedContainer);
	}

	@Override
	public Kind supportedKind() {
		return Kind.SHARED_MAILBOX;
	}
}
