/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2023
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

package net.bluemind.user.service.internal;

import java.util.ArrayList;
import java.util.List;

import net.bluemind.core.api.Email;
import net.bluemind.core.auditlogs.AuditLogUpdateStatus;
import net.bluemind.core.auditlogs.AuditLogUpdateStatus.MessageCriticity;
import net.bluemind.core.auditlogs.ContentElement;
import net.bluemind.core.auditlogs.ContentElement.ContentElementBuilder;
import net.bluemind.core.container.model.ChangeLogEntry.Type;
import net.bluemind.directory.service.AuditLogDirEntryMapper;
import net.bluemind.directory.service.DirEntryAndValue;
import net.bluemind.user.api.User;

public class AuditLogDirEntryUserMapper extends AuditLogDirEntryMapper<User> {

	@Override
	public ContentElement createContentElement(DirEntryAndValue<User> dirEntryAndValue, Type type) {
		ContentElementBuilder builder = super.createContentElementBuilder(dirEntryAndValue);
		builder.with(dirEntryAndValue.value.emails.stream().map(m -> m.toString()).toList());
		return builder.build();
	}

	@Override
	public AuditLogUpdateStatus createUpdateMessage(DirEntryAndValue<User> oldValue, DirEntryAndValue<User> newValue)
			throws Exception {
		StringBuilder sBuilder = new StringBuilder();
		AuditLogUpdateStatus auditLogUpdateStatus = new AuditLogUpdateStatus();
		if (!oldValue.entry.displayName.equals(newValue.entry.displayName)) {
			sBuilder.append("Changed displayname: '" + oldValue.entry.displayName + "' -> '"
					+ newValue.entry.displayName + "'" + CRLF);
		}
		if (!oldValue.entry.email.equals(newValue.entry.email)) {
			sBuilder.append(
					"Changed default email: '" + oldValue.entry.email + "' -> '" + newValue.entry.email + "'" + CRLF);
		}
		List<String> oldMails = new ArrayList<Email>(oldValue.mailbox.emails).stream().map(Email::toString).toList();
		List<String> newMails = new ArrayList<Email>(newValue.mailbox.emails).stream().map(Email::toString).toList();
		List<String> removed = oldMails.stream().filter(e -> !newMails.contains(e)).toList();
		removed.forEach(r -> sBuilder.append("removed email: " + r + CRLF));

		List<String> added = newMails.stream().filter(e -> !oldMails.contains(e)).toList();
		added.forEach(r -> sBuilder.append("added email: " + r + CRLF));

		if (!oldValue.value.accountType.equals(newValue.value.accountType)) {
			sBuilder.append(
					"Changed account type: " + oldValue.value.accountType + " -> " + newValue.value.accountType);
		}
		String quotaMessage = mailQuotaMessage(oldValue.mailbox.quota, newValue.mailbox.quota);
		if (!quotaMessage.isBlank()) {
			sBuilder.append(quotaMessage + CRLF);
		}
		String updateMessage = sBuilder.toString();
		auditLogUpdateStatus.updateMessage = updateMessage;
		if (updateMessage.isBlank()) {
			auditLogUpdateStatus.crit = MessageCriticity.MINOR;
		}
		return auditLogUpdateStatus;
	}

	private String mailQuotaMessage(Integer oldQuota, Integer newQuota) {
		if (oldQuota == null && newQuota != null && newQuota != 0) {
			return String.format("new quota added: %d kb", newQuota);
		}
		if (oldQuota != null && newQuota == null) {
			return String.format("quota has been removed");
		}
		if (oldQuota == null && (newQuota == null || newQuota == 0)) {
			return "";
		}
		if (oldQuota.intValue() != newQuota.intValue()) {
			return String.format("quota has changed: %d kb -> %d kb", oldQuota, newQuota);
		}
		return "";
	}

}
