/* BEGIN LICENSE
 * Copyright Â© Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.ui.gwtsharing.client;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.Collection;

import com.google.gwt.core.client.GWT;
import com.google.gwt.dom.client.Style.Cursor;
import com.google.gwt.dom.client.Style.Unit;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.resources.client.ClientBundle;
import com.google.gwt.resources.client.CssResource;
import com.google.gwt.uibinder.client.UiBinder;
import com.google.gwt.uibinder.client.UiField;
import com.google.gwt.user.client.ui.Button;
import com.google.gwt.user.client.ui.CheckBox;
import com.google.gwt.user.client.ui.DialogBox;
import com.google.gwt.user.client.ui.DockLayoutPanel;
import com.google.gwt.user.client.ui.FlexTable;
import com.google.gwt.user.client.ui.FlowPanel;
import com.google.gwt.user.client.ui.HTMLPanel;
import com.google.gwt.user.client.ui.HTMLTable.Cell;
import com.google.gwt.user.client.ui.Label;

import net.bluemind.calendar.api.IPublishCalendarPromise;
import net.bluemind.calendar.api.PublishMode;
import net.bluemind.calendar.api.gwt.endpoint.PublishCalendarGwtEndpoint;
import net.bluemind.core.container.api.IContainersPromise;
import net.bluemind.core.container.api.gwt.endpoint.ContainersGwtEndpoint;
import net.bluemind.core.container.model.acl.AccessControlEntry;
import net.bluemind.core.container.model.acl.Verb;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.IDirectoryAsync;
import net.bluemind.directory.api.gwt.endpoint.DirectorySockJsEndpoint;
import net.bluemind.gwtconsoleapp.base.handler.DefaultAsyncHandler;
import net.bluemind.ui.common.client.errors.ErrorCodeTexts;
import net.bluemind.ui.common.client.forms.Ajax;
import net.bluemind.ui.common.client.forms.CommonForm;
import net.bluemind.ui.common.client.forms.CrudConstants;
import net.bluemind.ui.common.client.forms.acl.AclAutoComplete;
import net.bluemind.ui.common.client.forms.acl.AclComboList;
import net.bluemind.ui.common.client.forms.acl.AclCombination;
import net.bluemind.ui.common.client.forms.acl.AclConstants;
import net.bluemind.ui.common.client.forms.acl.AclComboDelegationList;
import net.bluemind.ui.common.client.forms.acl.AclEntity;
import net.bluemind.ui.common.client.forms.acl.IEntitySelectTarget;
import net.bluemind.ui.common.client.icon.Trash;

public class AclEdit extends CommonForm implements IEntitySelectTarget {

	@UiField
	FlexTable table;

	@UiField
	FlowPanel publicComboContainer;

	@UiField
	FlowPanel publicAddressContainer;

	@UiField
	FlowPanel privateAddressContainer;

	@UiField
	FlowPanel externalContainer;

	@UiField
	AclAutoComplete autocomplete;

	@UiField
	Label noSharing;

	public static interface Resources extends ClientBundle {

		@Source("AclEdit.css")
		Style editStyle();

	}

	public static interface Style extends CssResource {

		String aclContainer();

		String name();

		String trash();

		String info();

		String warning();

		String warningIcon();

		String warningDialogContent();

	}

	private static final Resources res = GWT.create(Resources.class);
	private static final AclConstants constants = GWT.create(AclConstants.class);
	private static AclUiBinder uiBinder = GWT.create(AclUiBinder.class);
	private static final CrudConstants cc = GWT.create(CrudConstants.class);

	interface AclUiBinder extends UiBinder<HTMLPanel, AclEdit> {
	}

	private IDirectoryAsync directory;
	private final Style s;
	private CheckBox publicCheckbox;
	private AclComboList publicCombo;
	private AclComboDelegationList publicDelegationCombo;
	private Map<String, String> verbs;
	private Map<String, String> delegateVerbs;
	private HashMap<AclEntity, AclCombination> entities;
	public String domainUid;
	private String containerUid;
	private String containerType;
	private FlexTable pubAddress;
	private FlexTable privAddress;

	private AbstractDirEntryOpener opener;

	private List<IAclEntityValidator> validators = new ArrayList<>();

	public AclEdit(Map<String, String> verbs, Map<String, String> delegateVerbs, AbstractDirEntryOpener opener) {
		super();
		this.opener = opener;
		s = res.editStyle();
		s.ensureInjected();

		this.verbs = verbs;
		this.delegateVerbs = delegateVerbs;
		entities = new HashMap<AclEntity, AclCombination>();

		form = uiBinder.createAndBindUi(this);

		table.setVisible(false);
		noSharing.setVisible(false);

		this.publicDelegationCombo = new AclComboDelegationList(delegateVerbs);

		publicSharing();

		shareCalendarToExternal();

		table.setStyleName(s.aclContainer());

		autocomplete.setTarget(this);
		autocomplete.getElement().setId("acl-edit-autocomplete");
	}

	private void publicSharing() {
		publicCombo = new AclComboList(verbs);
		publicCombo.getElement().setId("acl-edit-public-combo");
		publicCombo.setEnable(false);

		publicCheckbox = new CheckBox(constants.aclAllowPublic());
		publicCheckbox.getElement().setId("acl-edit-public-checkbox");
		publicCheckbox.setValue(false);

		publicCheckbox.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				publicCombo.setEnable(publicCheckbox.getValue());
			}
		});
		FlexTable pub = new FlexTable();
		pub.setWidget(0, 0, publicCheckbox);
		pub.setWidget(1, 0, publicCombo);
		publicComboContainer.add(pub);
	}

	private void shareCalendarToExternal() {
		// Public address
		pubAddress = new FlexTable();
		Button pubButton = new Button(constants.allowPublicAddress());

		pubButton.addClickHandler(new ClickHandler() {

			@Override
			public void onClick(ClickEvent event) {
				IPublishCalendarPromise service = new PublishCalendarGwtEndpoint(Ajax.TOKEN.getSessionId(),
						containerUid).promiseApi();
				service.generateUrl(PublishMode.PUBLIC).thenRun(() -> reloadPublishedCalendarUrls());
			}
		});

		pubButton.getElement().setId("acl-edit-public-address-checkbox");
		pubAddress.setWidget(0, 0, pubButton);
		Label publicLabel = new Label(constants.publicAddressDesc());
		pubAddress.setWidget(1, 0, publicLabel);
		publicAddressContainer.add(pubAddress);

		// Private address
		privAddress = new FlexTable();
		Button privButton = new Button(constants.allowPrivateAddress());

		privButton.addClickHandler(new ClickHandler() {

			@Override
			public void onClick(ClickEvent event) {
				IPublishCalendarPromise service = new PublishCalendarGwtEndpoint(Ajax.TOKEN.getSessionId(),
						containerUid).promiseApi();
				service.generateUrl(PublishMode.PRIVATE).thenRun(() -> reloadPublishedCalendarUrls());
			}
		});

		privButton.getElement().setId("acl-edit-private-address-checkbox");
		privAddress.setWidget(0, 0, privButton);
		Label privLabel = new Label(constants.privateAddressDesc());
		privAddress.setWidget(1, 0, privLabel);
		privateAddressContainer.add(privAddress);
	}

	public void setAddressesSharing(String containerType) {
		externalContainer.setVisible(true);
	}

	public void setVerbs(Map<String, String> verbs, Map<String, String> delegateVerbs) {
		this.verbs = verbs;
		this.delegateVerbs = delegateVerbs;
		publicCombo.setVerbs(verbs);
		publicDelegationCombo.setVerbs(delegateVerbs);
	}

	public void setDomainUid(String domainUid) {
		directory = new DirectorySockJsEndpoint(Ajax.TOKEN.getSessionId(), domainUid);
		this.domainUid = domainUid;
		autocomplete.setDomain(domainUid);
	}

	@Override
	public void seletected(AclEntity aclEntity) {
		boolean present = false;
		for (AclEntity entry : entities.keySet()) {
			if (entry.getEntry().entryUid.equals(aclEntity.getEntry().entryUid)) {
				present = true;
				break;
			}
		}
		if (!present) {
			Optional<ValidationResult> error = validate(aclEntity);
			if (error.isPresent()) {
				aclEntityWarning(error.get().getErrorMessage());
			} else {
				addEntry(aclEntity);
			}
		}

	}

	private Optional<ValidationResult> validate(AclEntity aclEntity) {
		return validators.stream().map(validator -> validator.validate(aclEntity)).filter(result -> !result.isValid())
				.findFirst();
	}

	public void setValue(List<AccessControlEntry> acList) {
		if (directory == null) {
			throw new RuntimeException("domainUid is not defined");
		}

		table.removeAllRows();
		publicCombo.setEnable(false);
		publicCombo.setValue(Verb.Invitation); // default verb
		publicCheckbox.setValue(false);
		hasSharing(false);

		acList.stream() //
			.collect(Collectors.groupingBy(ace -> ace.subject)) //
			.entrySet().stream() //
			.forEach(entry -> {
			String aclSubject = entry.getKey();
			List<AccessControlEntry> aclForEntry = AccessControlEntry.compact(entry.getValue());
			if (aclSubject.equals(domainUid)) {
				boolean hasVerb = aclForEntry != null && !aclForEntry.isEmpty() && aclForEntry.get(0) != null;
				publicCombo.setValue(hasVerb ? aclForEntry.get(0).verb : null);
				publicCombo.setEnable(true);
				publicCheckbox.setValue(true);
			} else {
				directory.findByEntryUid(aclSubject, new DefaultAsyncHandler<DirEntry>() {
					@Override
					public void success(DirEntry value) {
						if (value != null) {
							addEntry(new AclEntity(value, aclForEntry));
						}
					}
				});
			}
		});

		reloadPublishedCalendarUrls();
	}

	private void reloadPublishedCalendarUrls() {
		IContainersPromise cservice = new ContainersGwtEndpoint(Ajax.TOKEN.getSessionId()).promiseApi();
		cservice.get(containerUid).thenAccept(container -> {
			if (container.type.equals("calendar")) {
				IPublishCalendarPromise service = new PublishCalendarGwtEndpoint(Ajax.TOKEN.getSessionId(),
						containerUid).promiseApi();
				loadUrls(service, PublishMode.PUBLIC, pubAddress);
				loadUrls(service, PublishMode.PRIVATE, privAddress);
			}
		});
	}
	private void loadUrls(IPublishCalendarPromise service, PublishMode mode, FlexTable tbl) {
		if (tbl.getRowCount() > 1) {
			for (int i = tbl.getRowCount() - 1; i >= 1; i--) {
				tbl.removeRow(i);
			}
		}

		service.getGeneratedUrls(mode).thenAccept(urls -> {
			for (int i = 0; i < urls.size(); i++) {
				Label url = new Label(urls.get(i));
				url.setHeight("35px");
				url.setStyleName("");
				tbl.setWidget(i + 1, 0, url);
				Button trash = new Button();
				trash.setStyleName("fa fa-trash-o");
				tbl.setWidget(i + 1, 1, trash);
				final int index = i;
				trash.addClickHandler(new ClickHandler() {
					@Override
					public void onClick(ClickEvent event) {
						service.disableUrl(urls.get(index)).thenRun(() -> reloadPublishedCalendarUrls());
					}
				});
			}
		});
	}

	public List<AccessControlEntry> getValue() {
		List<AclEntity> aclEntities = getEntries();
		List<AccessControlEntry> acl = new ArrayList<>(aclEntities.size());
		for (AclEntity entity : aclEntities) {
			acl.add(AccessControlEntry.create(entity.getEntry().entryUid, entity.getVerb()));
			if (entity.getDelVerb() != null) {
				acl.add(AccessControlEntry.create(entity.getEntry().entryUid, entity.getDelVerb()));
			}
		}
		return acl;
	}

	private void hasSharing(boolean hasSharing) {
		table.setVisible(hasSharing);
		noSharing.setVisible(!hasSharing);
	}

	private String iconStyleByKind(Kind kind) {
		String style = "fa fa-lg fa-user";
		switch (kind) {
			case GROUP:
				style = "fa fa-lg fa-users";
				break;
			case MAILSHARE:
				style = "fa fa-lg fa-inbox";
				break;
			case SHARED_MAILBOX:
				style = "fa fa-lg fa-envelope-square";
				break;
			case EXTERNALUSER:
				style = "fa fa-lg fa-user-secret";
				break;
			case RESOURCE:
				style = "fa fa-lg fa-briefcase";
				break;
			default:
				break;
		}
		return style;
	}

	private void addEntry(final AclEntity aclEntity) {
		final String key = aclEntity.getEntry().entryUid;

		AclComboList combo = new AclComboList(verbs);
		combo.getElement().setId("acl-edit-entry-" + key);

		if (combo.isValidValue(aclEntity.getVerb())) {
			boolean isMailboxAclCtx = "mailboxacl".equals(getContainerType());
			boolean isMeAndMyself = containerUid.endsWith(key);

			Label delegationMessage = new Label(constants.delegationMessage());
			delegationMessage.setVisible(false);
			AclComboDelegationList delegateCombo = new AclComboDelegationList(delegateVerbs);
			delegateCombo.getElement().setId("acl-del-edit-entry-" + key);
			AclCombination aclCombination = new AclCombination(combo, delegateCombo, delegationMessage, isMailboxAclCtx);

			Optional<ValidationResult> error = validate(aclEntity);
			int row = table.getRowCount();
			combo.setValue(aclEntity.getVerb());
			delegateCombo.setValue(isMailboxAclCtx ? aclEntity.getDelVerb() : null);
			
			Label icon = new Label();
			icon.setStyleName(iconStyleByKind(aclEntity.getEntry().kind));

			Trash trash = new Trash();
			trash.setId("acl-edit-trash-" + key);
			trash.setVisible(!isMeAndMyself);
			trash.addClickHandler(new ClickHandler() {
				@Override
				public void onClick(ClickEvent event) {
					entities.remove(aclEntity);
					Cell c = table.getCellForEvent(event);
					table.removeRow(c.getRowIndex());
					hasSharing(table.getRowCount() > 0);
				}
			});

			Label warning = new Label();
			warning.setStyleName("fa fa-lg fa-exclamation-circle");
			warning.addStyleName(s.warningIcon());
			warning.setVisible(false);

			entities.put(aclEntity, aclCombination);
			table.setWidget(row, 0, icon);

			Label name = new Label(aclEntity.getEntry().displayName);
			name.setTitle(aclEntity.getEntry().displayName);
			table.setWidget(row, 1, name);
			table.getCellFormatter().setStyleName(row, 1, s.name());

			combo.setEnable(!isMeAndMyself);
			table.setWidget(row, 2, combo);

			boolean allForMailbox = aclEntity.getVerb() == Verb.All;

			if (isMailboxAclCtx) {
				boolean delegateComboNotVisible = isMeAndMyself || aclEntity.getVerb() == Verb.All;
				aclCombination.updateDelegationPanelContent(!delegateComboNotVisible);

				table.setWidget(row, 3, aclCombination.delegationPanel);
				table.setWidget(row, 4, warning);
				table.setWidget(row, 5, trash);
				table.getCellFormatter().setStyleName(row, 4, s.warning());
				table.getCellFormatter().setStyleName(row, 5, s.trash());
			} else {
				table.setWidget(row, 3, warning);
				table.setWidget(row, 4, trash);
				table.getCellFormatter().setStyleName(row, 3, s.warning());
				table.getCellFormatter().setStyleName(row, 4, s.trash());
			}

			if (error.isPresent()) {
				warning.setTitle(error.get().getErrorMessage());
				warning.setVisible(true);
			}
			if (opener != null) {
				name.getElement().getStyle().setCursor(Cursor.POINTER);
				name.addClickHandler(e -> {
					opener.open(domainUid, aclEntity.getEntry());
				});

			}
		}

		hasSharing(!entities.isEmpty());
	}

	private List<AclEntity> getEntries() {
		List<AclEntity> aclEntities = new LinkedList<AclEntity>();

		// Public
		if (publicCheckbox.getValue()) {
			DirEntry dir = new DirEntry(); // FIXME public
			dir.entryUid = domainUid;
			AclEntity pub = new AclEntity(dir, publicCombo.getValue(), null);
			aclEntities.add(pub);
		}

		// Specific
		for (AclEntity aclEntity : entities.keySet()) {
			AclCombination aclEntityCombine = entities.get(aclEntity);
			aclEntities.add(new AclEntity(aclEntity.getEntry(), aclEntityCombine.getAclComboList().getValue(), aclEntityCombine.getAclComboDelegationList().getValue()));
		}

		return aclEntities;
	}

	/**
	 * @param items
	 * @param sb
	 */
	private void aclEntityWarning(String errorMessage) {
		DialogBox os = new DialogBox();

		FlowPanel buttons = new FlowPanel();
		Button ok = new Button(cc.done());
		ok.addStyleName("button");
		ok.addStyleName("primary");
		ok.addClickHandler(new ClickHandler() {

			@Override
			public void onClick(ClickEvent event) {
				os.hide();
			}
		});

		buttons.add(ok);
		buttons.getElement().getStyle().setPadding(5, Unit.PX);

		DockLayoutPanel dlp = new DockLayoutPanel(Unit.PX);
		dlp.setHeight("150px");
		dlp.setWidth("500px");

		Label warn = new Label(ErrorCodeTexts.INST.PERMISSION_DENIED());
		warn.addStyleName("modal-dialog-title");
		dlp.addNorth(warn, 40);

		Label content = new Label(errorMessage);
		content.addStyleName(s.warningDialogContent());
		dlp.addSouth(buttons, 40);

		dlp.add(content);

		os.addStyleName("dialog");
		os.getElement().setAttribute("style", "padding:0");
		os.setWidget(dlp);
		os.setGlassEnabled(true);
		os.setAutoHideEnabled(false);
		os.setGlassStyleName("modalOverlay");
		os.setModal(true);
		os.center();
		os.show();
	}

	public void setEnable(boolean e) {
		publicCheckbox.setEnabled(e);
		table.removeAllRows();
		autocomplete.setEnable(e);
	}

	public void setVisible(boolean b) {
		form.setVisible(b);
	}

	public void setContainerUid(String containerUid) {
		this.containerUid = containerUid;
	}

	public void setContainerType(String containerType) {
		this.containerType = containerType;
	}

	public String getContainerUid() {
		return this.containerUid;
	}
	
	public String getContainerType() {
		return this.containerType;
	}

	public void setPublicSharingVisible(boolean b) {
		publicComboContainer.setVisible(b);
	}

	public void registerValidator(IAclEntityValidator validator) {
		this.validators.add(validator);
	}

}