/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.ui.gwtsharing.client;

import java.util.List;
import java.util.ArrayList;
import java.util.Set;
import java.util.HashSet;
import java.util.stream.Collectors;

import com.google.gwt.core.client.JavaScriptObject;

import net.bluemind.core.api.AsyncHandler;
import net.bluemind.core.commons.gwt.JsMapStringJsObject;
import net.bluemind.core.container.api.IContainerManagementAsync;
import net.bluemind.core.container.api.IContainerManagementPromise;
import net.bluemind.core.container.api.gwt.endpoint.ContainerManagementGwtEndpoint;
import net.bluemind.core.container.model.acl.AccessControlEntry;
import net.bluemind.directory.api.IDirectoryPromise;
import net.bluemind.directory.api.gwt.endpoint.DirectoryGwtEndpoint;
import net.bluemind.gwtconsoleapp.base.editor.gwt.IGwtModelHandler;
import net.bluemind.gwtconsoleapp.base.handler.DefaultAsyncHandler;
import net.bluemind.ui.common.client.forms.Ajax;

public abstract class BaseSharingModelHandler implements IGwtModelHandler {

	private final String modelId;
	private List<AccessControlEntry> acl;

	public BaseSharingModelHandler(String modelId) {
		this.modelId = modelId;
	}

	@Override
	public void load(JavaScriptObject model, final AsyncHandler<Void> handler) {
		final JsMapStringJsObject m = model.cast();

		IContainerManagementPromise cm = new ContainerManagementGwtEndpoint(Ajax.TOKEN.getSessionId(),
				getContainerUid(model)).promiseApi();
		cm.getDescriptor().thenCompose(container -> {
			IDirectoryPromise directoryService = new DirectoryGwtEndpoint(Ajax.TOKEN.getSessionId(),
					container.domainUid).promiseApi();
			return directoryService.getEntry(container.ownerDirEntryPath);
		}).thenCompose(owner -> {
			SharingModel.init(m, modelId);
			SharingModel.populate(m, modelId, owner);
			return cm.getAccessControlList();
		}).thenAccept(value -> {
			BaseSharingModelHandler.this.acl = value;
			SharingModel.populate(m, modelId, value);
			handler.success(null);
		}).exceptionally(t -> {
			handler.success(null);
			return null;
		});

	}

	@Override
	public void save(JavaScriptObject model, final AsyncHandler<Void> handler) {
		SharingModel sm = SharingModel.get(model, modelId);
		if (sm == null) {
			handler.success(null);
		} else {

			List<AccessControlEntry> acl = sm.getAcl();
			String containerUid = getContainerUid(model);
			AclComparator comparator = new AclComparator(containerUid, this.acl, acl);

			if (!comparator.aclEquals()) {
				Set<AccessControlEntry> newTarget = comparator.getNewTarget();
				IContainerManagementAsync cm = new ContainerManagementGwtEndpoint(Ajax.TOKEN.getSessionId(),
						containerUid);

				cm.getAccessControlList(new DefaultAsyncHandler<List<AccessControlEntry>>() {

					@Override
					public void success(List<AccessControlEntry> value) {
						Set<AccessControlEntry> aclToAdd = value.stream().filter(ace -> ace.subject.startsWith("x-calendar-p")).collect(Collectors.toSet());

						Set<AccessControlEntry> aclToSet = new HashSet<>();
						aclToSet.addAll(newTarget);
						aclToSet.addAll(aclToAdd);

						cm.setAccessControlList(aclToSet.stream().collect(Collectors.toList()), handler);
					}

				});

			} else {
				handler.success(null);
				return;
			}
		}

	}

	protected abstract String getContainerUid(JavaScriptObject model);
}
