/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2025
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.ui.adminconsole.directory.sharedmailbox;

import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.gwt.core.client.GWT;
import com.google.gwt.core.client.JavaScriptObject;
import com.google.gwt.dom.client.Element;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.KeyUpEvent;
import com.google.gwt.json.client.JSONObject;
import com.google.gwt.uibinder.client.UiBinder;
import com.google.gwt.uibinder.client.UiField;
import com.google.gwt.uibinder.client.UiHandler;
import com.google.gwt.user.client.ui.CheckBox;
import com.google.gwt.user.client.ui.HTMLPanel;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.ListBox;
import com.google.gwt.user.client.ui.TextBox;

import net.bluemind.addressbook.api.gwt.js.JsVCard;
import net.bluemind.addressbook.api.gwt.js.JsVCardIdentification;
import net.bluemind.addressbook.api.gwt.js.JsVCardIdentificationName;
import net.bluemind.core.commons.gwt.JsMapStringJsObject;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.model.gwt.js.JsItemValue;
import net.bluemind.core.container.model.gwt.serder.ItemValueGwtSerDer;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.gwt.js.JsDomain;
import net.bluemind.domain.api.gwt.serder.DomainGwtSerDer;
import net.bluemind.gwtconsoleapp.base.editor.Ajax;
import net.bluemind.gwtconsoleapp.base.editor.WidgetElement;
import net.bluemind.gwtconsoleapp.base.editor.gwt.CompositeGwtWidgetElement;
import net.bluemind.gwtconsoleapp.base.editor.gwt.GwtWidgetElement;
import net.bluemind.gwtconsoleapp.base.editor.gwt.IGwtDelegateFactory;
import net.bluemind.gwtconsoleapp.base.editor.gwt.IGwtWidgetElement;
import net.bluemind.mailbox.api.gwt.js.JsMailboxRouting;
import net.bluemind.ui.adminconsole.base.DomainsHolder;
import net.bluemind.ui.adminconsole.base.SubscriptionInfoHolder;
import net.bluemind.ui.adminconsole.base.ui.DelegationEdit;
import net.bluemind.ui.adminconsole.directory.sharedmailbox.l10n.SharedMailboxConstants;
import net.bluemind.ui.common.client.errors.ErrorCodeTexts;
import net.bluemind.ui.common.client.forms.finder.ServerFinder;
import net.bluemind.user.api.gwt.js.JsUser;

public class NewSharedMailbox extends CompositeGwtWidgetElement {

	interface EditSharedMailboxUiBinder extends UiBinder<HTMLPanel, NewSharedMailbox> {
	}

	public static final String TYPE = "bm.ac.QCreateSharedMailboxWidget";

	private static EditSharedMailboxUiBinder uiBinder = GWT.create(EditSharedMailboxUiBinder.class);

	private ItemValue<Domain> domain;

	@UiField
	TextBox displayName;

	@UiField
	TextBox login;

	@UiField
	DelegationEdit delegation;

	@UiField
	CheckBox hidden;

	@UiField
	Label errorLabel;

	private NewSharedMailbox() {
		HTMLPanel dlp = uiBinder.createAndBindUi(this);

		initWidget(dlp);
		displayName.getElement().setId("new-shared-mailbox-displayname");
		login.getElement().setId("new-shared-mailbox-login");
		login.getElement().setAttribute("autocomplete", "new-shared-mailbox-login");
		login.getElement().setAttribute("login", "new-shared-mailbox-login");
		hidden.getElement().setId("new-shared-mailbox-hidden");
		updateDomainChange(DomainsHolder.get().getSelectedDomain());
	}

	@UiHandler("login")
	void loginKeyboard(KeyUpEvent e) {
		if (!login.getText().isEmpty()) {
			displayName.setText(login.getText());
		}
	}

	@Override
	public void loadModel(JavaScriptObject model) {
		JsMapStringJsObject map = model.cast();
		if (map.get("domain") != null) {
			JsItemValue<JsDomain> domain = map.get("domain").cast();

			ItemValue<Domain> d = new ItemValueGwtSerDer<>(new DomainGwtSerDer()).deserialize(new JSONObject(domain));
			updateDomainChange(d);
		}
	}

	@Override
	public void saveModel(JavaScriptObject model) {
		JsMapStringJsObject map = model.cast();

		JavaScriptObject t = map.get("sharedMailbox");
		JsUser sharedMailbox = null;
		if (t != null) {
			sharedMailbox = t.cast();
		} else {
			sharedMailbox = JsUser.create();
			map.put("sharedMailbox", sharedMailbox);
		}

		sharedMailbox.setLogin(login.getText());
		sharedMailbox.setLoginCaps(false);
		sharedMailbox.setPasswordMustChange(false);
		sharedMailbox.setPasswordNeverExpires(false);
		sharedMailbox.setHidden(hidden.getValue());
		sharedMailbox.setRouting(JsMailboxRouting.none());

		sharedMailbox.setOrgUnitUid(delegation.asEditor().getValue());
		JsVCard vcard = JsVCard.create();
		vcard.setIdentification(JsVCardIdentification.create());
		vcard.getIdentification().setPhoto(false);
		JsVCardIdentificationName vcardname = JsVCardIdentificationName.create();
		vcard.getIdentification().setName(vcardname);
		sharedMailbox.setContactInfos(vcard);

		vcardname.setGivenNames(login.getText());
		map.putString("accountType", "FULL");

	}

	@Override
	public void attach(Element parent) {
		super.attach(parent);
		login.setFocus(true);
	}

	private void updateDomainChange(ItemValue<Domain> active) {
		this.domain = active;
		delegation.setDomain(active.uid);
		if (domain.value.global) {
			errorLabel.setText(ErrorCodeTexts.INST.getString("NOT_IN_GLOBAL_DOMAIN"));
		} else {
			errorLabel.setText("");
		}
	}

	public static void registerType() {
		GwtWidgetElement.register(TYPE, new IGwtDelegateFactory<IGwtWidgetElement, WidgetElement>() {

			@Override
			public IGwtWidgetElement create(WidgetElement e) {
				return new NewSharedMailbox();
			}
		});
		GWT.log("bm.ac.QCreateSharedMailboxWidget registred");

	}

}
