/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.todolist.service.internal;

import java.sql.SQLException;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;

import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.IContainerManagement;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.api.ItemValueExists;
import net.bluemind.core.container.model.BaseContainerDescriptor;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.ContainerModifiableDescriptor;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.model.acl.Verb;
import net.bluemind.core.container.persistence.ContainerStore;
import net.bluemind.core.container.persistence.ContainerSyncStore;
import net.bluemind.core.container.persistence.DataSourceRouter;
import net.bluemind.core.container.service.internal.ItemValueAuditLogService;
import net.bluemind.core.container.service.internal.RBACManager;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.sanitizer.Sanitizer;
import net.bluemind.core.task.api.TaskRef;
import net.bluemind.core.task.service.BlockingServerTask;
import net.bluemind.core.task.service.IServerTask;
import net.bluemind.core.task.service.IServerTaskMonitor;
import net.bluemind.core.task.service.ITasksManager;
import net.bluemind.core.validator.Validator;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.lib.elasticsearch.ESearchActivator;
import net.bluemind.todolist.api.ITodoListsMgmt;
import net.bluemind.todolist.api.ITodoUids;
import net.bluemind.todolist.api.VTodo;
import net.bluemind.todolist.persistence.VTodoIndexStore;
import net.bluemind.todolist.persistence.VTodoStore;
import net.bluemind.todolist.service.IInCoreTodoListsMgmt;

public class TodoListsMgmt implements ITodoListsMgmt, IInCoreTodoListsMgmt {
	static final Logger logger = LoggerFactory.getLogger(TodoListsMgmt.class);
	private BmContext context;
	private RBACManager rbacManager;
	private Validator validator;
	private Sanitizer sanitizer;

	public TodoListsMgmt(BmContext context) {
		this.context = context;
		rbacManager = new RBACManager(context);
		sanitizer = new Sanitizer(context);
		validator = new Validator(context);
	}

	@Override
	public TaskRef reindexAll() throws ServerFault {
		return context.provider().instance(ITasksManager.class).run(reindexAllTask());
	}

	@Override
	public void reindexAll(IServerTaskMonitor monitor) throws Exception {
		reindexAllTask().execute(monitor);
	}

	@Override
	public TaskRef reindex(String calUid) throws ServerFault {
		return context.provider().instance(ITasksManager.class).run(reindexTask(calUid));

	}

	@Override
	public void reindex(String calUid, IServerTaskMonitor monitor) throws Exception {
		reindexTask(calUid).execute(monitor);
	}

	private IServerTask reindexAllTask() {
		return new BlockingServerTask() {

			@Override
			public void run(IServerTaskMonitor monitor) throws Exception {
				if (!context.getSecurityContext().isDomainGlobal()) {
					throw new ServerFault("only admin0 can call this method ", ErrorCode.FORBIDDEN);
				}

				Set<String> all = getContainerUids();

				monitor.begin(all.size() + 1, "begin todolists reindexation [" + all.size() + "]");
				ESearchActivator.resetIndex("todo");
				monitor.progress(1, "Index todo reseted");

				for (String uid : all) {
					IServerTaskMonitor subMonitor = monitor.subWork("todolist [" + uid + "]", 1);
					try {
						reindex(uid, subMonitor);
					} catch (ServerFault sf) {
						logger.error("Failed to reindex todolist {}: {}", uid, sf.getMessage());
						monitor.log("Failed to reindex todolist " + uid);
					}

				}
			}
		};
	}

	private IServerTask reindexTask(final String todolistUid) {
		return new BlockingServerTask() {

			@Override
			public void run(IServerTaskMonitor monitor) throws Exception {
				DataSource ds = DataSourceRouter.get(context, todolistUid);
				ContainerStore containerStore = new ContainerStore(context, ds, context.getSecurityContext());

				Container c = containerStore.get(todolistUid);

				if (c == null) {
					throw ServerFault.notFound("todolist " + todolistUid + " not found");
				}

				if (!context.getSecurityContext().isDomainAdmin(c.domainUid)) {
					throw new ServerFault("only admin of " + c.domainUid + " can call this method ",
							ErrorCode.FORBIDDEN);
				}

				reindex(c, monitor);
			}
		};
	}

	private void reindex(Container container, IServerTaskMonitor monitor) throws ServerFault {
		DataSource ds = DataSourceRouter.get(context, container.uid);

		BaseContainerDescriptor containerDescriptor = BaseContainerDescriptor.create(container.uid, container.name,
				container.owner, container.type, container.domainUid, container.defaultContainer);
		containerDescriptor.internalId = container.id;
		ItemValueAuditLogService<VTodo> logService = new ItemValueAuditLogService<>(context, containerDescriptor);
		VTodoContainerStoreService storeService = new VTodoContainerStoreService(context, ds,
				context.getSecurityContext(), container, new VTodoStore(ds, container), logService);

		VTodoIndexStore indexStore = new VTodoIndexStore(ESearchActivator.getClient(), container,
				DataSourceRouter.location(context, container.uid));
		logger.info("reindexing todolist {}", container.uid);
		// reinit container index
		indexStore.deleteAll();

		List<String> uids = storeService.allUids();
		monitor.begin(uids.size() + 1, "reindexing todolist [" + container.uid + "] (size:" + uids.size() + ")");
		Lists.partition(uids, 500).forEach(subUids -> {
			List<ItemValue<VTodo>> values = storeService.getMultiple(subUids);
			indexStore.updates(values);
			monitor.progress(subUids.size(), "todolist [" + container.uid + "] reindexing...");
		});

		// only report one time
		monitor.progress(1, "todolist [" + container.uid + "] indexed");
		logger.info("todolist {} reindexed", container.uid);

	}

	private Set<String> getContainerUids() throws SQLException {
		Collection<DataSource> dataSources = context.getAllMailboxDataSource();
		Set<String> all = new LinkedHashSet<>();

		for (DataSource ds : dataSources) {
			ContainerStore cs = new ContainerStore(context, ds, context.getSecurityContext());
			List<Container> containers = cs.findByType(ITodoUids.TYPE);
			all.addAll(containers.stream().map(c -> c.uid).collect(Collectors.toList()));
		}

		ContainerStore cs = new ContainerStore(context, context.getDataSource(), context.getSecurityContext());
		List<Container> containers = cs.findByType(ITodoUids.TYPE);
		all.addAll(containers.stream().map(c -> c.uid).collect(Collectors.toList()));

		return all;
	}

	@Override
	public void delete(String uid) {
		ContainerDescriptor descriptor = getComplete(uid);

		if (descriptor == null) {
			throw ServerFault.notFound("todolist " + uid + " not found");
		}

		DataSource ds = DataSourceRouter.get(context, uid);
		ContainerStore containerStore = new ContainerStore(context, ds, context.getSecurityContext());
		Container container;
		try {
			container = containerStore.get(uid);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}

		BaseContainerDescriptor containerDescriptor = BaseContainerDescriptor.create(container.uid, container.name,
				container.owner, container.type, container.domainUid, container.defaultContainer);
		containerDescriptor.internalId = container.id;
		ItemValueAuditLogService<VTodo> logService = new ItemValueAuditLogService<>(context, containerDescriptor);
		VTodoContainerStoreService storeService = new VTodoContainerStoreService(context, ds,
				context.getSecurityContext(), container, new VTodoStore(ds, container), logService);

		storeService.prepareContainerDelete();
		context.su().provider().instance(IContainers.class).delete(uid);
	}

	@Override
	public ContainerDescriptor get(String uid) {
		return getComplete(uid);
	}

	@Override
	public void restore(ItemValue<ContainerDescriptor> item, boolean isCreate) {
		if (isCreate) {
			create(item, false);
		} else {
			update(item);
		}
	}

	@Override
	public ItemValueExists itemValueExists(String uid) {
		return new ItemValueExists(getComplete(uid) != null, true);
	}

	@Override
	public void create(String uid, ContainerDescriptor descriptor, boolean isDefault) throws ServerFault {
		ItemValue<ContainerDescriptor> item = ItemValue.create(uid, descriptor);
		create(item, isDefault);
	}

	private void create(ItemValue<ContainerDescriptor> item, boolean isDefault) throws ServerFault {
		String uid = item.uid;
		ContainerDescriptor descriptor = item.value;
		sanitizer.create(descriptor);
		validator.create(descriptor);
		IDirectory dir = context.provider().instance(IDirectory.class, descriptor.domainUid);
		DirEntry entry = dir.findByEntryUid(descriptor.owner);
		if (entry == null) {
			throw ServerFault.notFound("owner " + descriptor.owner + " not found in domain " + descriptor.domainUid);
		}

		checkCanManageTodo(descriptor, entry.kind);

		ContainerDescriptor cd = ContainerDescriptor.create(uid, descriptor.name, descriptor.owner, ITodoUids.TYPE,
				descriptor.domainUid, isDefault);
		if (descriptor.settings.getOrDefault("readonly", "false").equals("true")) {
			cd.readOnly = true;
		}

		IContainers containers = context.su().provider().instance(IContainers.class);
		containers.create(cd.uid, cd);

		if (!descriptor.settings.isEmpty()) {
			IContainerManagement containerManagement = context.provider().instance(IContainerManagement.class, cd.uid);
			containerManagement.setSettings(descriptor.settings);
		}

		DataSource ds = DataSourceRouter.get(context, cd.uid);
		ContainerStore cs = new ContainerStore(context, ds, SecurityContext.SYSTEM);
		Container container = null;
		try {
			container = cs.get(cd.uid);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}

		ContainerSyncStore syncStore = new ContainerSyncStore(ds, container);
		syncStore.initSync();

	}

	private void checkCanManageTodo(ContainerDescriptor descriptor, Kind ownerKind) throws ServerFault {
		if (ownerKind != Kind.USER && ownerKind != Kind.SHARED_MAILBOX) {
			throw new ServerFault("Invalid owner " + ownerKind, ErrorCode.INVALID_PARAMETER);
		}

		if (!rbacManager.forDomain(descriptor.domainUid).forEntry(descriptor.owner).can(Verb.Manage.name())
				&& (!(context.getSecurityContext().getSubject().equals(descriptor.owner)
						&& context.getSecurityContext().getContainerUid().equals(descriptor.domainUid)))) {
			throw new ServerFault("cannot manage this todolist", ErrorCode.PERMISSION_DENIED);
		}

	}

	@Override
	public ContainerDescriptor getComplete(String uid) throws ServerFault {
		IContainers containers = context.provider().instance(IContainers.class);
		ContainerDescriptor cd = containers.getIfPresent(uid);
		if (cd == null) {
			return null;
		}

		if (cd.type.equals(ITodoUids.TYPE)) {
			return cd;
		}

		logger.warn("trying to retrieve a todolist descriptor but it's not an todolist but a {}", cd.type);
		return null;
	}

	@Override
	public void update(String uid, ContainerDescriptor descriptor) throws ServerFault {
		ItemValue<ContainerDescriptor> item = ItemValue.create(uid, descriptor);
		update(item);
	}

	private void update(ItemValue<ContainerDescriptor> item) throws ServerFault {
		String uid = item.uid;
		ContainerDescriptor descriptor = item.value;
		ContainerDescriptor old = getComplete(uid);

		if (old == null) {
			throw ServerFault.notFound("todolist " + uid + " not found");
		}

		sanitizer.update(old, descriptor);
		validator.update(old, descriptor);

		IDirectory dir = context.provider().instance(IDirectory.class, old.domainUid);
		DirEntry entry = dir.findByEntryUid(old.owner);
		if (entry == null) {
			throw ServerFault.notFound("owner " + old.owner + " not found in domain " + old.domainUid);
		}

		if (!old.owner.equals(descriptor.owner)) {
			throw new ServerFault("trying to change todolist owner", ErrorCode.INVALID_PARAMETER);
		}

		ContainerModifiableDescriptor cmd = new ContainerModifiableDescriptor();
		cmd.name = descriptor.name;
		context.su().provider().instance(IContainers.class).update(uid, cmd);

	}

}
