/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.system.service.internal;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import org.ini4j.Ini;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.datastax.oss.driver.shaded.guava.common.base.Suppliers;

import net.bluemind.core.api.ParametersValidator;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.service.internal.RBACManager;
import net.bluemind.core.rest.BmContext;
import net.bluemind.repository.provider.RepositoryProvider;
import net.bluemind.role.api.BasicRoles;
import net.bluemind.system.api.ISystemConfiguration;
import net.bluemind.system.api.SystemConf;
import net.bluemind.system.api.SystemState;
import net.bluemind.system.repository.ISystemConfStore;
import net.bluemind.system.service.hooks.SystemConfigurationHooks;
import net.bluemind.system.state.StateContext;

public class SystemConfiguration implements ISystemConfiguration {
	private static final Logger logger = LoggerFactory.getLogger(SystemConfiguration.class);

	private static final String INIFILE = "/etc/bm/bm.ini";

	private Supplier<ISystemConfStore> systemConfStore;
	private BmContext context;
	private RBACManager rbac;
	private final Supplier<Optional<SystemConfigurationAuditLogService>> auditLogSupplier;

	public SystemConfiguration(BmContext context) {
		this.context = context;
		this.systemConfStore = Suppliers.memoize(() -> RepositoryProvider.instance(ISystemConfStore.class, context));
		rbac = new RBACManager(context);

		auditLogSupplier = () -> {
			if (StateContext.getState().equals(SystemState.CORE_STATE_RUNNING)) {
				var auditLog = new SystemConfigurationAuditLogService(context, "sysconf");
				return Optional.of(auditLog);
			} else {
				return Optional.empty();
			}
		};

	}

	@Override
	public SystemConf getValues() throws ServerFault {
		rbac.check(BasicRoles.ROLE_MANAGE_SYSTEM_CONF);

		return SystemConf
				.create(getDbValues().entrySet().stream().map(SystemConfigurationHooks.getInstance()::obfuscateValue)
						.collect(Collectors.toMap(Entry::getKey, Entry::getValue)));
	}

	@Override
	public void updateMutableValues(Map<String, String> v) throws ServerFault {
		rbac.check(BasicRoles.ROLE_MANAGE_SYSTEM_CONF);
		ParametersValidator.notNull(v);

		Map<String, String> values = new HashMap<>(v);

		SystemConf previous = SystemConf.create(getDbValues());
		SystemConfigurationHooks.getInstance().sanitize(previous, values);
		SystemConfigurationHooks.getInstance().validate(previous, values);
		Map<String, String> merged = SystemConf.merge(previous, values);

		ServerFault.onExceptionVoid(() -> systemConfStore.get().update(merged), ErrorCode.SQL_ERROR);

		auditLogSupplier.get().ifPresent(auditlog -> auditlog.logUpdate(merged, previous.values));
		SystemConfigurationHooks.getInstance().fireUpdated(context, previous, SystemConf.create(merged));
	}

	private Map<String, String> getDbValues() {
		Map<String, String> values = new HashMap<>();
		File iniFile = new File(INIFILE);
		if (iniFile.exists()) {
			try (InputStream in = Files.newInputStream(iniFile.toPath())) {
				values = new Ini(in).get("global").entrySet().stream()
						.collect(Collectors.toMap(Entry::getKey, Entry::getValue));
			} catch (IOException e) {
				logger.error("Error during loading {}: {}", INIFILE, e.getMessage());
			}
		} else if (logger.isDebugEnabled()) {
			logger.debug("{} not found", INIFILE);
		}

		try {
			values.putAll(systemConfStore.get().get());
		} catch (Exception e) {
			logger.warn("error retrieving system configuration from database ({}), will try 3.0 bm-info",
					e.getMessage());

			try {
				values.putAll(systemConfStore.get().get30());
			} catch (Exception e1) {
				if (new File("/etc/bm/bm-core.tok").exists()) {
					// Something goes wrong
					// BlueMind is already installed but we cannot
					// retrieve configuration from database
					throw new ServerFault("Fail to fetch system configuration", e1);
				}

				logger.error("error retrieving system configuration (3.0) from database: {}", e1.getMessage());
			}
		}

		return values;
	}

}
