/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2025
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.system.service.internal;

import java.sql.SQLException;
import java.util.List;
import java.util.Objects;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.api.ListResult;
import net.bluemind.core.api.ParametersValidator;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.DataLocation;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.persistence.ItemStore;
import net.bluemind.core.container.service.internal.RBACManager;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.core.tx.wrapper.TxEnabler;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.IInCoreDomains;
import net.bluemind.domain.service.property.Obfuscate;
import net.bluemind.hornetq.client.Shared;
import net.bluemind.repository.provider.RepositoryProvider;
import net.bluemind.role.api.BasicRoles;
import net.bluemind.system.api.Credential;
import net.bluemind.system.api.IInternalCredentials;
import net.bluemind.system.api.KafkaCredentialEntry;
import net.bluemind.system.api.SysConfKeys;
import net.bluemind.system.repository.ICredentialsStore;
import net.bluemind.tx.outbox.api.ITxOutbox;

public class CredentialsService implements IInternalCredentials {
	private static final Logger logger = LoggerFactory.getLogger(CredentialsService.class);

	private final String domainUid;
	private final RBACManager rbac;
	private final ICredentialsStore credentialsStore;
	private final ItemStore itemStore;
	private BmContext context;

	private final Obfuscate propertyObfuscator = new Obfuscate();
	private final IInCoreDomains inCoreDomainsService = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM)
			.instance(IInCoreDomains.class);

	public CredentialsService(BmContext context, String domainUid, Container container) {
		this.domainUid = domainUid;
		this.context = context;
		this.rbac = new RBACManager(context).forDomain(container.uid);

		this.itemStore = new ItemStore(context.getDataSource(), container, context.getSecurityContext());
		this.credentialsStore = RepositoryProvider.instance(ICredentialsStore.class, context);

	}

	@Override
	public String getDomainCredentialById(String credentialId) {
		rbac.check(BasicRoles.ROLE_SYSTEM_MANAGER);
		ParametersValidator.notNullAndNotEmpty(credentialId);

		if (logger.isDebugEnabled()) {
			logger.debug("Getting credential {} for domain {}", credentialId, domainUid);
		}

		if (domainUid.equals("global.virt")) {
			return getGlobalCredential(credentialId);
		}

		return getDomainCredential(credentialId);
	}

	private String getDomainCredential(String credentialId) {
		// Support only obfuscated domain properties
		if (!propertyObfuscator.isObfuscated(credentialId)) {
			return null;
		}

		ItemValue<Domain> domain = inCoreDomainsService.getUnfiltered(domainUid);
		if (domain == null) {
			return null;
		}

		return domain.value.properties.get(credentialId);
	}

	private String getGlobalCredential(String credentialId) {
		if (!SysConfKeys.obfuscateOnGet(credentialId)) {
			// Can get only obfuscated values from sysconf
			return null;
		}

		return Shared.mapSysconf().get(credentialId);
	}

	@Override
	public void addUserCredential(String userUid, Credential credential) {
		rbac.forDomain(domainUid).forEntry(userUid).check(BasicRoles.ROLE_MANAGE_USER_PASSWORD);

		long userId = getUserIdFromUid(userUid);
		TxEnabler.atomically(() -> {
			try {
				credentialsStore.create(userId, credential);

				ITxOutbox outbox = context.su().provider().instance(ITxOutbox.class, domainUid, userUid,
						KafkaCredentialEntry.TYPE, userUid, DataLocation.directory().serverUid());
				outbox.forKafka(ItemValue.create(credential.id, new KafkaCredentialEntry(userUid, credential)), null,
						false);
			} catch (SQLException e) {
				throw ServerFault.sqlFault(e);
			}
		});

	}

	@Override
	public List<Credential> getUserCredentials(String userUid) {
		rbac.check(BasicRoles.ROLE_SYSTEM_MANAGER);

		return getPlainUserCredentials(userUid);
	}

	private List<Credential> getPlainUserCredentials(String userUid) {
		long userId = getUserIdFromUid(userUid);
		try {
			return credentialsStore.get(userId);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	@Override
	public void removeUserCredential(String userUid, String credentialId) {
		rbac.forDomain(domainUid).forEntry(userUid).check(BasicRoles.ROLE_MANAGE_USER_PASSWORD);

		long userId = getUserIdFromUid(userUid);
		TxEnabler.atomically(() -> {
			try {
				credentialsStore.delete(userId, credentialId);

				ITxOutbox outbox = context.su().provider().instance(ITxOutbox.class, domainUid, userUid,
						KafkaCredentialEntry.TYPE, userUid, DataLocation.directory().serverUid());
				Credential credential = new Credential();
				credential.id = credentialId;
				outbox.forKafka(ItemValue.create(credentialId, new KafkaCredentialEntry(userUid, credential)), null,
						true);
			} catch (SQLException e) {
				throw ServerFault.sqlFault(e);
			}
		});
	}

	private long getUserIdFromUid(String userUid) {
		try {
			Long itemId = itemStore.getItemId(userUid);
			if (itemId == null) {
				throw new ServerFault("User UID: " + userUid + " not found in domain: " + domainUid,
						ErrorCode.NOT_FOUND);
			}

			return itemId.longValue();
		} catch (SQLException e) {
			throw new ServerFault("Unable to get user item from UID: " + userUid, e);
		}
	}

	@Override
	public void addUserCredentials(String userUid, List<Credential> credentials) {
		rbac.check(BasicRoles.ROLE_SYSTEM_MANAGER);

		long userId = getUserIdFromUid(userUid);
		try {
			credentialsStore.batchInsert(userId, credentials);
		} catch (SQLException e) {
			throw new ServerFault(e);
		}
	}

	@Override
	public ListResult<Credential> getObfuscatedUserCredentials(String userUid) {
		rbac.forEntry(userUid).check(BasicRoles.ROLE_MANAGE_USER_PASSWORD);

		List<Credential> userCredentials = getPlainUserCredentials(userUid).stream().filter(Objects::nonNull)
				.map(uc -> {
					uc.credential = null;
					uc.secret = null;
					return uc;
				}).toList();

		return ListResult.create(userCredentials);
	}

	@Override
	public void restoreUserCredential(String userUid, Credential credential) {
		rbac.check(BasicRoles.ROLE_SYSTEM_MANAGER);
		try {
			long userId = getUserIdFromUid(userUid);
			credentialsStore.create(userId, credential);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	@Override
	public void deleteUserCredential(String userUid, String credentialId) {
		rbac.check(BasicRoles.ROLE_SYSTEM_MANAGER);
		try {
			long userId = getUserIdFromUid(userUid);
			credentialsStore.delete(userId, credentialId);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}
}
