/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2023
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

package net.bluemind.system.service.internal;

import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.MapDifference;
import com.google.common.collect.Maps;

import net.bluemind.core.auditlogs.AuditLogEntry;
import net.bluemind.core.auditlogs.AuditLogUpdateStatus;
import net.bluemind.core.auditlogs.AuditLogUpdateStatus.MessageCriticity;
import net.bluemind.core.container.model.ChangeLogEntry.Type;
import net.bluemind.core.container.service.internal.AuditLogService;
import net.bluemind.core.rest.BmContext;

public class SystemConfigurationAuditLogService extends AuditLogService<Map<String, String>, Map<String, String>> {

	private static final Logger logger = LoggerFactory.getLogger(SystemConfigurationAuditLogService.class);
	private static final String EXTERNAL_URL_KEY = "external-url";

	public SystemConfigurationAuditLogService(BmContext ctx, String type) {
		super(ctx, type);
		domainUid = ctx.getSecurityContext().getContainerUid();
	}

	@Override
	protected AuditLogEntry createAuditLogEntry(Map<String, String> map, Type type) {
		AuditLogEntry auditLogEntry = new AuditLogEntry();
		auditLogEntry.logtype = type();
		auditLogEntry.content = mapper.createContentElement(map, type);
		auditLogEntry.securityContext = createSecurityContextElement(ctx.getSecurityContext());
		auditLogEntry.action = type.name();
		return auditLogEntry;
	}

	@Override
	protected AuditLogUpdateStatus createUpdateStatus(Map<String, String> newValues, Map<String, String> oldValues) {
		MapDifference<String, String> diff = Maps.difference(oldValues, newValues);
		StringBuilder sb = new StringBuilder();

		diff.entriesOnlyOnLeft().forEach((k, v) -> {
			// external-url not allowed in DB -> must always be removed (see
			// ExternalUrlHook)
			if (!k.equals(EXTERNAL_URL_KEY)) {
				sb.append(String.format("Removed conf '%s'='%s'", k, v));
				sb.append("\r\n");
			}
		});
		diff.entriesOnlyOnRight().forEach((k, v) -> {
			sb.append(String.format("Added conf '%s'='%s'", k, v));
			sb.append("\r\n");
		});
		diff.entriesDiffering().forEach((k, v) -> {
			sb.append(String.format("Updated value for key '%s' : '%s' -> '%s'", k, v.leftValue(), v.rightValue()));
			sb.append("\r\n");
		});
		return new AuditLogUpdateStatus(sb.toString());
	}

	@Override
	public void logUpdate(Map<String, String> newValues, Map<String, String> oldValues) {
		AuditLogUpdateStatus updateStatus = createUpdateStatus(newValues, oldValues);
		AuditLogEntry auditLogEntry = createAuditLogEntry(newValues, Type.Updated);
		auditLogEntry.domainUid = domainUid;
		auditLogEntry.updatemessage = updateStatus.updateMessage;
		auditLogEntry.criticity = MessageCriticity.MAJOR;
		store(auditLogEntry);
	}

}
