/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.system.ldap.importation.hooks;

import java.util.Map;
import java.util.Optional;

import org.apache.directory.api.ldap.model.cursor.CursorException;
import org.apache.directory.api.ldap.model.cursor.EntryCursor;
import org.apache.directory.api.ldap.model.exception.LdapException;
import org.apache.directory.api.ldap.model.message.BindResponse;
import org.apache.directory.api.ldap.model.message.ResultCodeEnum;
import org.apache.directory.api.ldap.model.message.SearchScope;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.domain.api.Domain;
import net.bluemind.system.importation.commons.Parameters;
import net.bluemind.system.importation.commons.UuidMapper;
import net.bluemind.system.importation.commons.hooks.ImportAuthenticationService;
import net.bluemind.system.importation.commons.pool.LdapConnectionContext;
import net.bluemind.system.ldap.importation.api.LdapConstants;
import net.bluemind.system.ldap.importation.api.LdapProperties;
import net.bluemind.system.ldap.importation.internal.tools.LdapParameters;
import net.bluemind.system.ldap.importation.internal.tools.LdapUuidMapper;
import net.bluemind.system.ldap.importation.metrics.LdapMetricsHolder;
import net.bluemind.system.ldap.importation.search.LdapUserSearchFilter;

public class ImportLdapAuthenticationService extends ImportAuthenticationService {
	private static final Logger logger = LoggerFactory.getLogger(ImportLdapAuthenticationService.class);

	public ImportLdapAuthenticationService() {
		super(LdapMetricsHolder.get());
	}

	@Override
	protected String getDirectoryKind() {
		return "LDAP";
	}

	@Override
	protected String getPrefix() {
		return LdapConstants.EXTID_PREFIX;
	}

	@Override
	protected Parameters getParameters(Domain domain, Map<String, String> domainSettings, String credential) {
		return LdapParameters.build(domain, domainSettings, credential);
	}

	@Override
	protected Optional<UuidMapper> getUuidMapper(String externalId) {
		return LdapUuidMapper.fromExtId(externalId);
	}

	@Override
	protected String searchUserDnFromLogin(LdapConnectionContext ldapConCtx, String userLogin)
			throws LdapException, CursorException {
		EntryCursor result = ldapConCtx.ldapCon.search(ldapConCtx.parameters.ldapDirectory.baseDn,
				new LdapUserSearchFilter().getSearchFilterByName(ldapConCtx.parameters, userLogin), SearchScope.SUBTREE,
				"dn");

		if (!result.next()) {
			return null;
		}

		return result.get().getDn().getName();
	}

	@Override
	protected String searchUserDnFromUuid(LdapConnectionContext ldapConCtx, String uuid)
			throws LdapException, CursorException {
		String filter = new LdapUserSearchFilter().getSearchFilterByUuid(ldapConCtx.parameters, uuid);
		EntryCursor result = ldapConCtx.ldapCon.search(ldapConCtx.parameters.ldapDirectory.baseDn, filter,
				SearchScope.SUBTREE, "dn");

		if (!result.next()) {
			return null;
		}

		return result.get().getDn().getName();
	}

	@Override
	protected AuthResult bindResponseToAuthResult(Parameters parameters, LdapConnectionContext ldapConCtx,
			BindResponse bindResponse, long directoryBindDuration, String userDn) {
		if (ResultCodeEnum.SUCCESS != bindResponse.getLdapResult().getResultCode()
				|| !ldapConCtx.ldapCon.isAuthenticated()) {
			if (logger.isDebugEnabled()) {
				logger.debug(
						"Login failed on: {}:{}:{}, result: {}, message: {}, authenticated: {}, user dn: {},ldapAuth: {}ms",
						ldapConCtx.getConnectedProtocol().name(), ldapConCtx.ldapConnectionConfig.getLdapHost(),
						ldapConCtx.ldapConnectionConfig.getLdapPort(),
						bindResponse.getLdapResult().getResultCode().name(),
						bindResponse.getLdapResult().getDiagnosticMessage(), ldapConCtx.ldapCon.isAuthenticated(),
						userDn, directoryBindDuration);
			}

			return AuthResult.NO;
		}

		if (logger.isInfoEnabled()) {
			logger.info("Login success on: {}:{}:{}, user dn: {}, ldapAuth: {}ms",
					ldapConCtx.getConnectedProtocol().name(), ldapConCtx.ldapConnectionConfig.getLdapHost(),
					ldapConCtx.ldapConnectionConfig.getLdapPort(), userDn, directoryBindDuration);
		}

		return AuthResult.YES;
	}

	@Override
	protected String getDirectoryCredentialPropertyName() {
		return LdapProperties.import_ldap_password.name();
	}
}
