/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.system.importation.commons.pool;

import java.io.IOException;
import java.time.Duration;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import org.apache.directory.api.ldap.codec.api.ConfigurableBinaryAttributeDetector;
import org.apache.directory.api.ldap.codec.api.DefaultConfigurableBinaryAttributeDetector;
import org.apache.directory.api.ldap.model.exception.LdapException;
import org.apache.directory.api.ldap.model.exception.LdapOperationException;
import org.apache.directory.api.ldap.model.exception.LdapTlsHandshakeException;
import org.apache.directory.api.ldap.model.message.ResultCodeEnum;
import org.apache.directory.ldap.client.api.DefaultPoolableLdapConnectionFactory;
import org.apache.directory.ldap.client.api.LdapConnection;
import org.apache.directory.ldap.client.api.LdapConnectionConfig;
import org.apache.directory.ldap.client.api.LdapConnectionPool;
import org.apache.directory.ldap.client.api.NoVerificationTrustManager;
import org.apache.directory.ldap.client.api.exception.LdapConnectionTimeOutException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.util.concurrent.RateLimiter;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.lib.ldap.LdapProtocol;
import net.bluemind.system.importation.commons.Parameters;
import net.bluemind.system.importation.commons.Parameters.Server.Host;

public class LdapPoolWrapper {
	@SuppressWarnings("serial")
	private static class StartTlsFault extends ServerFault {
	}

	private static final Logger logger = LoggerFactory.getLogger(LdapPoolWrapper.class);
	private static final long LDAP_TIMEOUT = TimeUnit.SECONDS.toMillis(2);
	private static final int POOL_UNAVAILABILITY = 10;
	private static final long POOL_UNAVAILABILITY_MS = TimeUnit.SECONDS.toMillis(POOL_UNAVAILABILITY);
	// Unavailable pool for POOL_UNAVAILABILITYs if 2 LDAP connections timeout/s
	private static final int CONNECTION_TIMEOUT_PER_SECONDS = 2;
	private static final RateLimiter connectionTimeoutLimiter = RateLimiter.create(CONNECTION_TIMEOUT_PER_SECONDS);
	// Unavailable pool for POOL_UNAVAILABILITYs if 5 LDAP errors/s
	private static final int ERRORS_PER_SECONDS = 5;
	private static final RateLimiter errorLimiter = RateLimiter.create(ERRORS_PER_SECONDS);
	// Only 1 pool unavailable log/5s
	private static final RateLimiter ldapPoolUnavailableLogLimiter = RateLimiter.create(0.2);

	private LdapConnectionPool pool;
	private Parameters parameters;
	private LdapConnectionConfig connectionConfig;

	private Long unavailableUntil = System.currentTimeMillis() - POOL_UNAVAILABILITY_MS;

	public LdapPoolWrapper(Parameters parameters) {
		this.parameters = parameters;
	}

	public Optional<LdapConnectionContext> getConnection() {
		LdapConnection ldapCon = null;

		LdapConnectionPool currentPool = getPool();
		if (currentPool == null) {
			return Optional.empty();
		}

		try {
			ldapCon = currentPool.getConnection();
			return Optional.of(new LdapConnectionContext(ldapCon, connectionConfig, parameters));
		} catch (LdapConnectionTimeOutException lcte) {
			logger.error("Unable to get LDAP connection: {}", lcte.getMessage());

			if (!connectionTimeoutLimiter.tryAcquire()) {
				logger.warn("More than {} connections timeout/s", CONNECTION_TIMEOUT_PER_SECONDS);
				setPoolUnavailability();
			}
		} catch (RuntimeException | LdapException e) {
			if (e.getCause() instanceof InterruptedException) {
				logger.error("Getting an interrupted exception, reseting pool for {}", parameters, e);
				resetPool();
				return Optional.empty();
			}

			if (logger.isDebugEnabled()) {
				logger.debug("Unable to get LDAP connection", e);
			} else {
				logger.error("Unable to get LDAP connection: {}", e.getMessage());
			}

			if (!errorLimiter.tryAcquire()) {
				logger.warn("More than {} errors/s", ERRORS_PER_SECONDS);
				setPoolUnavailability();
			}
		}

		return Optional.empty();
	}

	private synchronized LdapConnectionPool getPool() {
		if (!isAvailable()) {
			if (ldapPoolUnavailableLogLimiter.tryAcquire()) {
				logger.warn("LDAP pool disabled for {}, retry later...", parameters);
			}

			return null;
		}

		if (pool == null) {
			initPoolFromHosts();

			if (pool == null) {
				logger.warn("No LDAP server available for {}, disabling pool for {}s", parameters,
						TimeUnit.MILLISECONDS.toSeconds(LdapPoolWrapper.POOL_UNAVAILABILITY_MS));
				setPoolUnavailability();
				return null;
			}

			logger.info("Connected to LDAP: {}", connectionConfig.getLdapHost());
		}

		return pool;
	}

	public boolean isAvailable() {
		return System.currentTimeMillis() > unavailableUntil;
	}

	private synchronized void setPoolUnavailability() {
		logger.warn("Suspend pool for {}", parameters);
		unavailableUntil = System.currentTimeMillis() + POOL_UNAVAILABILITY_MS;
	}

	private void initPoolFromHosts() {
		List<Host> ldapHosts = parameters.ldapServer.getLdapHost();
		if (ldapHosts == null || ldapHosts.isEmpty()) {
			return;
		}

		Iterator<Host> ldapHostsIterator = ldapHosts.iterator();
		while (pool == null && ldapHostsIterator.hasNext()) {
			Host ldapHost = ldapHostsIterator.next();

			connectionConfig = setLdapConnectionConfig(parameters, ldapHost);

			try {
				tryConnection(ldapHost, connectionConfig);
			} catch (StartTlsFault stf) {
				logger.error("Unable to connect tls:{}:{}: {}", ldapHost.hostname, ldapHost.port, stf.getMessage());

				if (parameters.ldapServer.protocol == LdapProtocol.TLSPLAIN) {
					connectionConfig.setUseTls(false);
					tryConnection(ldapHost, connectionConfig);
				}
			}
		}
	}

	private void tryConnection(Host ldapHost, LdapConnectionConfig ldapConnectionConfig) {
		if (logger.isInfoEnabled()) {
			logger.info("Trying to connect to: {}", getLdapConnectionURI(ldapHost, ldapConnectionConfig));
		}

		DefaultPoolableLdapConnectionFactory bpcf = new DefaultPoolableLdapConnectionFactory(ldapConnectionConfig);

		LdapConnectionPool tmpPool = null;
		LdapConnection conn = null;

		try {
			tmpPool = new LdapConnectionPool(bpcf);
			tmpPool.setMaxWait(Duration.ofSeconds(10));

			conn = tmpPool.getConnection();
			tmpPool.releaseConnection(conn);

			pool = tmpPool;
			this.connectionConfig = ldapConnectionConfig;
		} catch (LdapException e) {
			logger.warn("Unable to connect to {}: {}", ldapHost.hostname, e.getMessage());

			try {
				tmpPool.close();
			} catch (Exception ignored) {
				// ok
			}

			if (ldapConnectionConfig.isUseTls()
					&& ((e instanceof LdapOperationException loe && (loe.getResultCode() == ResultCodeEnum.UNAVAILABLE))
							|| (e instanceof LdapTlsHandshakeException))) {
				throw new StartTlsFault();
			}
		}
	}

	private String getLdapConnectionURI(Host ldapHost, LdapConnectionConfig ldapConnectionConfig) {
		String hostPort = ldapHost.hostname + ":" + ldapHost.port;

		if (ldapConnectionConfig.isUseSsl()) {
			return "ssl:" + hostPort;
		}

		return (ldapConnectionConfig.isUseTls() ? "tls:" : "plain:") + hostPort;
	}

	private LdapConnectionConfig setLdapConnectionConfig(Parameters ldapParameters, Host ldapHost) {
		LdapConnectionConfig config = new LdapConnectionConfig();
		config.setLdapHost(ldapHost.hostname);
		config.setLdapPort(ldapHost.port);
		config.setTimeout(LDAP_TIMEOUT);

		switch (ldapParameters.ldapServer.protocol) {
		case TLS, TLSPLAIN:
			config.setUseTls(true);
			config.setUseSsl(false);
			break;
		case SSL:
			config.setUseTls(false);
			config.setUseSsl(true);
			break;
		default:
			config.setUseTls(false);
			config.setUseSsl(false);
			break;
		}

		if (ldapParameters.ldapServer.acceptAllCertificates) {
			config.setTrustManagers(new NoVerificationTrustManager());
		}

		ConfigurableBinaryAttributeDetector detector = new DefaultConfigurableBinaryAttributeDetector();
		config.setBinaryAttributeDetector(detector);

		return config;
	}

	public void doReleaseConnection(LdapConnectionContext ldapConCtx) {
		if (ldapConCtx == null || ldapConCtx.ldapCon == null) {
			return;
		}

		if (pool == null) {
			logger.warn("No LDAP connection pool for: {}, closing connection", parameters);
			closeLdapConnection(ldapConCtx.ldapCon);
			return;
		}

		try {
			if (ldapConCtx.isError()) {
				logger.warn("Invalidate LDAP connection from pool {}", ldapConCtx.parameters);
				pool.invalidateObject(ldapConCtx.ldapCon);
				return;
			}

			if (ldapConCtx.ldapCon.isAuthenticated()) {
				ldapConCtx.ldapCon.anonymousBind();
			}

			pool.releaseConnection(ldapConCtx.ldapCon);
		} catch (Exception e) {
			logger.error("Unable to release connection from pool {}", ldapConCtx.parameters, e);
			resetPool();
		}
	}

	private void closeLdapConnection(LdapConnection ldapCon) {
		try {
			ldapCon.close();
		} catch (IOException ioe) {
			// No pool found, and unable to close orphan LDAP connection...
		}
	}

	private synchronized void resetPool() {
		logger.info("Reset LDAP pool for domain: {}", parameters);
		LdapConnectionPool poolToReset = pool;
		pool = null;

		if (poolToReset == null) {
			logger.warn("No LDAP connection pool for: {}", parameters);
			return;
		}

		try {
			poolToReset.clear();
			poolToReset.close();
		} catch (Exception e) {
			logger.error("Fail to close LDAP pool for: " + parameters.toString(), e);
		}
	}
}
