<?php

/* BEGIN LICENSE
 * Copyright Â© Blue Mind SAS, 2012-2017
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
namespace BM;
require_once 'GlobalEnv.php';

/**
 * Implementation of net.bluemind.system.api.IInternalCredentials.
 */

class InternalCredentialsClient {

  protected $base;
  protected $sid;
  protected $domainUid;

  /*
   * Constructor.
   *
   * @param base
   * @param sid
   * @param domainUid
   *
   */
  public function __construct($base, $sid , $domainUid) {
    $this->sid = $sid;
    $this->base = $base."/api/credentials/{domainUid}";
    $this->domainUid = $domainUid;
    $this->base = str_replace("{domainUid}", urlencode($domainUid), $this->base);
  }


  /*
   * @param totpCredential
   * @return
   */
  public function addTotpCredential( $totpCredential  ) {
    $postUri = "/user/totp";
    $method = "PUT";


    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    $body = $totpCredential;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param userUid
   * @param credential
   * @return
   */
  public function addUserCredential( $userUid ,  $credential  ) {
    $postUri = "/user/{userUid}";
    $method = "PUT";

    $postUri = str_replace("{userUid}", urlencode($userUid), $postUri);

    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    $body = $credential;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param userUid
   * @param credentials
   * @return
   */
  public function addUserCredentials( $userUid ,  $credentials  ) {
    $postUri = "/user/{userUid}/_credentials";
    $method = "PUT";

    $postUri = str_replace("{userUid}", urlencode($userUid), $postUri);

    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    $body = $credentials;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param credentialId
   * @return
   */
  public function getDomainCredentialById( $credentialId  ) {
    $postUri = "/{credentialId}";
    $method = "GET";

    $postUri = str_replace("{credentialId}", urlencode($credentialId), $postUri);

    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param userUid
   * @return
   */
  public function getObfuscatedUserCredentials( $userUid  ) {
    $postUri = "/user/{userUid}";
    $method = "GET";

    $postUri = str_replace("{userUid}", urlencode($userUid), $postUri);

    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @return
   */
  public function getSelfObfuscatedCredentials() {
    $postUri = "/user/_credentials/_self";
    $method = "GET";


    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param userUid
   * @return
   */
  public function getUserCredentials( $userUid  ) {
    $postUri = "/user/{userUid}/plain";
    $method = "GET";

    $postUri = str_replace("{userUid}", urlencode($userUid), $postUri);

    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param credentialId
   * @return
   */
  public function removeSelfCredential( $credentialId  ) {
    $postUri = "/user/_credentials/{credentialId}/_self";
    $method = "DELETE";

    $postUri = str_replace("{credentialId}", urlencode($credentialId), $postUri);

    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param userUid
   * @param credentialId
   * @return
   */
  public function removeUserCredential( $userUid ,  $credentialId  ) {
    $postUri = "/user/{userUid}/{credentialId}";
    $method = "DELETE";

    $postUri = str_replace("{userUid}", urlencode($userUid), $postUri);
    $postUri = str_replace("{credentialId}", urlencode($credentialId), $postUri);

    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param totpCredential
   * @return
   */
  public function validateTotpCredential( $totpCredential  ) {
    $postUri = "/user/totp/validate";
    $method = "POST";


    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    $body = $totpCredential;
    return $this->execute($url, $queryParam, $body, $method);
  }


  /*
   * Execute the request
   *
   * @param url
   * @param data
   * @param body
   */
  private function execute($url, $queryParam, $body, $method) {

    $curl = curl_init();

    $headers = array();
    array_push($headers, 'X-BM-ApiKey: '.$this->sid);

    if (sizeof($queryParam) > 0) {
      $url .= '?'.http_build_query($queryParam);
    }

    curl_setopt_array($curl, GlobalEnv::$curlOptions + array(
      CURLOPT_URL => $url,
      CURLOPT_HTTPHEADER => $headers,
      CURLOPT_CUSTOMREQUEST => $method)
    );

    if ($method == 'POST') {
      curl_setopt($curl, CURLOPT_POST, TRUE);
      curl_setopt($curl, CURLOPT_POSTFIELDS, array());
    }

    if (is_resource($body)) {
      if ($method == 'PUT') {
        curl_setopt($curl, CURLOPT_PUT, TRUE);
      }
      $size = fstat($body)['size'];
      curl_setopt($curl, CURLOPT_INFILE, $body);
      curl_setopt($curl, CURLOPT_INFILESIZE, $size);
    } else {
      if (is_object($body) && method_exists($body, 'serialize')) {
        $body = $body->serialize();
      } else if (is_object($body)) {
        $body = json_encode($body);
      } else if (is_array($body)) {
        $body = json_encode($body);
      } else if (is_string($body)) {
        $body = json_encode($body);
      }
      $size = strlen($body);
      array_push($headers, 'Content-Type: application/json');
      array_push($headers, 'Content-Length: '.$size);
      curl_setopt($curl, CURLOPT_HTTPHEADER, $headers);
      curl_setopt($curl, CURLOPT_POSTFIELDS, $body);
    }

    $resp = curl_exec($curl);
    if ($resp === false) {
      throw new \Exception(curl_error($curl));
    } 
    curl_close($curl);
    if (!$resp) {
      return;
    }
    $js = json_decode($resp);
    if ($js === NULL) {
      return $resp;
    }
    if (isset($js->errorCode)) {
      throw new \Exception($js->errorCode . ': ' . (isset($js->message) ? ' : ' . $js->message : ''));
    }
    return $js;
  }

}
