/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2023
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.sds.sync.service;

import java.util.List;
import java.util.concurrent.atomic.AtomicReference;

import io.vertx.core.AbstractVerticle;
import io.vertx.core.Verticle;
import io.vertx.core.eventbus.EventBus;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.lib.vertx.IUniqueVerticleFactory;
import net.bluemind.lib.vertx.IVerticleFactory;
import net.bluemind.sds.sync.api.SdsSyncEvent;
import net.bluemind.sds.sync.api.SdsSyncEvent.Body;
import net.bluemind.sds.sync.service.internal.queue.SdsSyncQueue;

public class SdsSyncVerticle extends AbstractVerticle {
	public static final String MANAGE_QUEUE_BUSNAME = "sds.sync.body.reset";
	public static final String RESET_EVENT = "RESET_EVENT";

	private AtomicReference<SdsSyncQueue> queue = new AtomicReference<>();

	public static class Factory implements IVerticleFactory, IUniqueVerticleFactory {
		@Override
		public boolean isWorker() {
			return true;
		}

		@Override
		public Verticle newInstance() {
			return new SdsSyncVerticle();
		}
	}

	@Override
	public void start() {
		EventBus bus = vertx.eventBus();
		queue.set(new SdsSyncQueue());

		bus.<Body>consumer(SdsSyncEvent.BODYADD.busName(), msg -> {
			queue.get().putBody(SdsSyncEvent.BODYADD, msg.body());
			msg.reply("ok");
		});
		bus.<List<Body>>consumer(SdsSyncEvent.BODYADDBATCH.busName(), msg -> {
			queue.get().putBodies(SdsSyncEvent.BODYADD, msg.body());
			msg.reply("ok");
		});
		bus.<Body>consumer(SdsSyncEvent.BODYDEL.busName(), msg -> {
			queue.get().putBody(SdsSyncEvent.BODYDEL, msg.body());
			msg.reply("ok");
		});
		bus.<String>consumer(SdsSyncEvent.FHADD.busName(), msg -> {
			queue.get().putFileHosting(SdsSyncEvent.FHADD, msg.body());
			msg.reply("ok");
		});

		vertx.eventBus().<String>consumer(MANAGE_QUEUE_BUSNAME, msg -> {
			if (RESET_EVENT.equals(msg.body())) {
				try {
					queue.get().reset();
				} catch (Exception e) {
					msg.fail(500, "Not able to reset sds queue");
					throw new ServerFault("Not able to reset sds queue", e);
				} finally {
					queue.set(new SdsSyncQueue());
				}
			} else {
				msg.fail(500, "Unexpected value: " + msg.body());
				throw new IllegalArgumentException("Unexpected value: " + msg.body());
			}
			msg.reply("ok");
		});
	}

	@Override
	public void stop() {
		try {
			queue.get().close();
		} catch (Exception e) {
			// Ignore
		}
	}

}