/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2023
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.sds.sync.service.internal.stream;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.netty.buffer.ByteBufUtil;
import io.netty.util.concurrent.DefaultThreadFactory;
import io.vertx.core.Context;
import io.vertx.core.Handler;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.json.JsonObject;
import io.vertx.core.streams.ReadStream;
import net.bluemind.core.api.Stream;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.sds.sync.api.SdsSyncEvent;
import net.bluemind.sds.sync.service.SdsSyncLock;
import net.bluemind.sds.sync.service.internal.RocksDBLookup;
import net.bluemind.sds.sync.service.internal.queue.SdsSyncQueue;
import net.openhft.chronicle.queue.ExcerptTailer;

public class CQSdsSyncReadStream implements ReadStream<Buffer>, Stream {
	private static final Logger logger = LoggerFactory.getLogger(CQSdsSyncReadStream.class);
	private static final ExecutorService SSEXECUTOR = Executors
			.newSingleThreadExecutor(new DefaultThreadFactory("sds-sync-cq-stream"));
	private static final int DEFAULT_SLICES_SLOT_SIZE = 512;

	private final SdsSyncQueue q;
	private Handler<Throwable> exceptionHandler;
	private final AtomicBoolean paused = new AtomicBoolean(true);
	private final AtomicBoolean ended = new AtomicBoolean(false);
	private final AtomicLong lastIndex = new AtomicLong(0L);
	private final AtomicLong currentIndex = new AtomicLong(0L);
	private final ExcerptTailer tailer;
	private final RocksDBLookup dbLookup;
	private Handler<Buffer> handler;
	private Handler<Void> endHandler;
	private final Context context;
	private List<JsonObject> slices;
	private final int slotSliceSize;

	public CQSdsSyncReadStream(Context context, long fromIndex, RocksDBLookup dbLookup) {
		this(context, fromIndex, dbLookup, DEFAULT_SLICES_SLOT_SIZE);
	}

	public CQSdsSyncReadStream(Context context, long fromIndex, RocksDBLookup dbLookup, int slotSize) {
		this.context = context;
		Objects.requireNonNull(context);
		q = new SdsSyncQueue();
		this.tailer = getTailer(fromIndex);
		this.dbLookup = dbLookup;
		this.slices = Collections.synchronizedList(new ArrayList<>(slotSize));
		this.slotSliceSize = slotSize;
	}

	private static <T> CompletableFuture<T> onCqThread(Supplier<T> r) {
		return CompletableFuture.supplyAsync(r, SSEXECUTOR).exceptionally(t -> {
			logger.error("failure on CQ thread", t);
			return null;
		});
	}

	private ExcerptTailer getTailer(long offset) {
		try {
			return onCqThread(() -> {
				ExcerptTailer t = q.createTailer();
				lastIndex.set(q.queue().lastIndex());
				if (offset > 0) {
					t.moveToIndex(offset);
				} else {
					t.toStart();
				}
				return t;
			}).get(5, TimeUnit.SECONDS);
		} catch (Exception e) { // NOSONAR
			throw ServerFault.create(ErrorCode.TIMEOUT, e);
		}
	}

	@Override
	public CQSdsSyncReadStream exceptionHandler(Handler<Throwable> handler) {
		this.exceptionHandler = handler;
		return this;
	}

	@Override
	public CQSdsSyncReadStream handler(Handler<Buffer> handler) {
		this.handler = handler;
		return this;
	}

	@Override
	public CQSdsSyncReadStream pause() {
		paused.set(true);
		return this;
	}

	private void end() {
		if (ended.compareAndSet(false, true)) {
			if (endHandler != null) {
				context.runOnContext(v -> endHandler.handle(null));
			}
			close();
		}
	}

	private void read() {
		if (paused.get() || ended.get()) {
			return;
		}
		fetchPending();
		if (slices.isEmpty() && currentIndex.get() >= lastIndex.get()) {
			end();
		}
	}

	private void fetchPending() {
		if (handler == null || ended.get()) {
			return;
		}
		if (SdsSyncLock.get().isLocked()) {
			throw new ServerFault("Sds sync rebuild in progress, please retry in a few minutes");
		}

		if (!slices.isEmpty()) {
			handleFirstSliceData();
			context.runOnContext(w -> read());
			return;
		}

		context.executeBlocking(this::populateSlices, true).andThen(ar -> {
			if (ar.failed()) {
				if (exceptionHandler != null) {
					context.runOnContext(v -> exceptionHandler.handle(ar.cause()));
				} else {
					logger.error("no exception handler for {}", ar.cause().getMessage(), ar.cause());
				}
			} else if (slices.isEmpty()) {
				end();
			} else {
				context.runOnContext(v -> read());
			}
		});

	}

	private void handleFirstSliceData() {
		JsonObject data = slices.removeFirst();
		if (data == null) {
			end();
		} else {
			if (Boolean.FALSE.equals(data.getBoolean("deleted", false))) {
				data.remove("deleted");
				Buffer buf = Buffer.buffer(data.encode());
				try {
					context.runOnContext(v -> handler.handle(buf));
				} catch (Exception ise) {
					// Pipe closed, set handler to null
					logger.error("IllegalStateException received {}: setting handler to null (request canceled?)",
							ise.getMessage());
					end();
				}
			}
		}
	}

	private Boolean populateSlices() throws InterruptedException, ExecutionException, TimeoutException {
		if (tailer.isClosed() || tailer.isClosing()) {
			logger.error("Refused to fetch: tailer {} is closing", tailer);
			throw new ServerFault("tailer is closing or closed");
		}

		return onCqThread(() -> {
			try {
				readDocuments();
			} catch (Throwable t) {
				logger.error("unknown error", t);
				throw t;
			} finally {
				currentIndex.set(tailer.index());
			}
			return !slices.isEmpty();
		}).get(5, TimeUnit.SECONDS);
	}

	private void readDocuments() {
		AtomicInteger readIndex = new AtomicInteger(0);
		while (readIndex.intValue() < slotSliceSize && //
				tailer.readDocument(r -> r.read("sdssync").marshallable(m -> {
					String type = m.read("type").text();
					JsonObject jo = new JsonObject();
					jo.put("type", type);

					if (SdsSyncEvent.FHADD.name().equals(type)) {
						jo.put("key", m.read("key").text());
					} else {
						byte[] bodyGuid = m.read("key").bytes();
						if (type.equals(SdsSyncEvent.BODYADD.name())) {
							jo.put("deleted", dbLookup.exists(bodyGuid));
						}
						jo.put("key", ByteBufUtil.hexDump(bodyGuid));
						jo.put("srv", m.read("srv").text());
					}
					jo.put("index", tailer.index());
					slices.add(jo);
					readIndex.incrementAndGet();
				})))
			;
	}

	@Override
	public CQSdsSyncReadStream resume() {
		paused.set(false);
		read();
		return this;
	}

	@Override
	public CQSdsSyncReadStream fetch(long amount) {
		return this;
	}

	@Override
	public CQSdsSyncReadStream endHandler(Handler<Void> endHandler) {
		this.endHandler = endHandler;
		return this;
	}

	private void close() {
		pause();
		try {
			q.close();
		} catch (Exception e) {
			logger.error("error while closing", e);
			if (exceptionHandler != null) {
				context.runOnContext(v -> exceptionHandler.handle(e));
			}
		}
		if (dbLookup != null) {
			try {
				dbLookup.close();
			} catch (Exception e) {
				// Ignored
			}
		}
	}
}