/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.sds.sync.service.internal;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Comparator;
import java.util.function.Supplier;
import java.util.stream.Stream;

import org.rocksdb.BlockBasedTableConfig;
import org.rocksdb.BloomFilter;
import org.rocksdb.CompressionType;
import org.rocksdb.FlushOptions;
import org.rocksdb.IndexType;
import org.rocksdb.LRUCache;
import org.rocksdb.Options;
import org.rocksdb.RocksDB;
import org.rocksdb.RocksDBException;
import org.rocksdb.WALRecoveryMode;
import org.rocksdb.WriteOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Suppliers;

public class RocksDBLookup implements AutoCloseable {
	private static final Logger logger = LoggerFactory.getLogger(RocksDBLookup.class);
	private static final byte[] DUMMY_VALUE = new byte[0];

	@SuppressWarnings("resource")
	// This is required because we must do loadLibrary() before accessing any
	// RocksDB things
	private static final Supplier<WriteOptions> writeOptions = Suppliers
			.memoize(() -> new WriteOptions().setDisableWAL(true).setSync(false));
	@SuppressWarnings("resource")
	private static final Supplier<FlushOptions> flushOptions = Suppliers
			.memoize(() -> new FlushOptions().setWaitForFlush(true));
	private final RocksDB db;
	private final Path dbPath;

	@SuppressWarnings("serial")
	public static class RocksDBLookupException extends RuntimeException {
		public RocksDBLookupException(Throwable e) {
			super(e);
		}
	}

	public RocksDBLookup(Path dbPath) throws RocksDBLookupException {
		RocksDB.loadLibrary();
		this.dbPath = dbPath;
		try {
			Files.createDirectories(dbPath);
		} catch (IOException ie) {
			logger.error("Unable to create rocksdb directory in {}: {}", dbPath, ie.getMessage());
			throw new RocksDBLookupException(ie);
		}

		BlockBasedTableConfig tableConfig = new BlockBasedTableConfig().setBlockSize(4 * 1024L) // 4KB
				.setBlockCache(new LRUCache(4 * 1024 * 1024L)) // 4MB cache
				.setCacheIndexAndFilterBlocks(true) //
				.setPinL0FilterAndIndexBlocksInCache(true) //
				.setFilterPolicy(new BloomFilter(10, false)) //
				.setIndexType(IndexType.kBinarySearch);

		@SuppressWarnings("resource")
		Options options = new Options() //
				.setCreateIfMissing(true) //
				.setWriteBufferSize(16 * 1024 * 1024L) //
				.setTableFormatConfig(tableConfig) //
				.setBestEffortsRecovery(true) //
				.setCompressionType(CompressionType.NO_COMPRESSION) // more disk, but less CPU
				.setWalRecoveryMode(WALRecoveryMode.SkipAnyCorruptedRecords);
		try {
			this.db = RocksDB.open(options, dbPath.toString());
		} catch (RocksDBException e) {
			logger.error("Unable to open rocksdb: {}", e.getMessage());
			throw new RocksDBLookupException(e);
		}

	}

	public void put(byte[] sha1Hash) throws RocksDBLookupException {
		if (sha1Hash == null || sha1Hash.length != 20) {
			throw new IllegalArgumentException("NO: not a sha1 hash");
		}
		try {

			db.put(writeOptions.get(), sha1Hash, DUMMY_VALUE);
		} catch (RocksDBException rdbe) {
			throw new RocksDBLookupException(rdbe);
		}
	}

	public void remove(byte[] sha1Hash) throws RocksDBLookupException {
		if (sha1Hash == null || sha1Hash.length != 20) {
			throw new IllegalArgumentException("NO: not a sha1 hash");
		}
		try {
			if (db.keyMayExist(sha1Hash, null)) {
				db.delete(writeOptions.get(), sha1Hash);
			}
		} catch (RocksDBException rdbe) {
			throw new RocksDBLookupException(rdbe);
		}
	}

	public void flush() {
		try {
			db.flush(flushOptions.get());
		} catch (RocksDBException rdbe) {
			throw new RocksDBLookupException(rdbe);
		}
	}

	public boolean exists(byte[] sha1Hash) {
		if (sha1Hash == null || sha1Hash.length != 20) {
			throw new IllegalArgumentException("NO: not a sha1 hash");
		}
		return db.keyExists(sha1Hash);
	}

	public void close() {
		db.close();
		try {
			try (Stream<Path> qstream = Files.walk(dbPath)) {
				qstream.sorted(Comparator.reverseOrder()).map(Path::toFile).forEach(File::delete);
			}
		} catch (IOException ie) {
			// Cleanup failed: ignored
		}
	}
}
