/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.retry.support.keydb;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.annotations.VisibleForTesting;

import io.lettuce.core.RedisClient;
import io.lettuce.core.api.sync.RedisCommands;
import net.bluemind.keydb.common.ClientProvider;
import net.bluemind.retry.support.common.RetryQueue;

public class KeydbQueue extends RetryQueue implements AutoCloseable {

	private static final Logger logger = LoggerFactory.getLogger(KeydbQueue.class);
	private final RedisClient redis;
	private final RedisCommands<String, String> rdCmd;
	private final String retryQ;
	private final String retryDLQ;
	private final String namespace;

	public KeydbQueue(String namespace) {
		this.namespace = namespace;
		this.redis = ClientProvider.newClient();
		this.rdCmd = this.redis.connect().sync();
		this.retryQ = "bm-core:retry:" + namespace;
		this.retryDLQ = retryQ + ":dlq";
	}

	private void pushTask(String tsk) {
		rdCmd.rpush(retryQ, tsk);
	}

	@Override
	public Appender writer() {
		return this::pushTask;
	}

	private record RedisRec(String payload) implements QueueRecord {
	}

	@Override
	public Tailer reader() {
		return new Tailer() {

			QueueRecord lastRead;

			@Override
			public QueueRecord next() {
				String payload = rdCmd.rpoplpush(retryQ, retryDLQ);
				if (payload != null) {
					lastRead = new RedisRec(payload);
				} else {
					lastRead = null;
				}
				return lastRead;
			}

			@Override
			public void commit() {
				if (lastRead != null) {
					rdCmd.lrem(retryDLQ, 1, lastRead.payload());
				}
			}
		};
	}

	/**
	 * We re-inject the dead letter queue into the main queue to be reprocessed on
	 * next flush round
	 */
	@Override
	public void compact() {
		String moved;
		int movedBack = 0;
		do {
			moved = rdCmd.rpoplpush(retryDLQ, retryQ);
			if (moved != null) {
				movedBack++;
			} else {
				break;
			}
		} while (true);
		if (movedBack > 0) {
			logger.warn("[{}] compaction moved {} item(s) for re-processing", namespace, movedBack);
		}
	}

	@Override
	public void close() {
		redis.close();
	}

	/**
	 * Remove all entries in the queue, used in JUnits
	 */
	@Override
	@VisibleForTesting
	public void clear() {
		logger.info("CLEARING queues {} and {}", retryQ, retryDLQ);
		rdCmd.del(retryQ);
		rdCmd.del(retryDLQ);
	}

}
