/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2023
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.retry.support;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Stopwatch;
import com.google.common.base.Suppliers;
import com.google.common.util.concurrent.RateLimiter;
import com.netflix.spectator.api.Counter;
import com.netflix.spectator.api.Timer;

import io.vertx.core.AbstractVerticle;
import io.vertx.core.Context;
import io.vertx.core.DeploymentOptions;
import io.vertx.core.Future;
import io.vertx.core.ThreadingModel;
import io.vertx.core.eventbus.EventBus;
import io.vertx.core.eventbus.Message;
import io.vertx.core.eventbus.MessageConsumer;
import io.vertx.core.json.JsonObject;
import net.bluemind.lib.vertx.VertxContext;
import net.bluemind.lib.vertx.VertxPlatform;
import net.bluemind.lifecycle.helper.SoftReset;
import net.bluemind.metrics.registry.IdFactory;
import net.bluemind.metrics.registry.MetricsRegistry;
import net.bluemind.retry.support.common.RetryQueue;
import net.bluemind.retry.support.common.RetryQueue.QueueRecord;
import net.bluemind.retry.support.common.RetryQueue.Tailer;
import net.bluemind.retry.support.keydb.KeydbQueue;

public class RetryQueueVerticle extends AbstractVerticle {

	private static final Logger logger = LoggerFactory.getLogger(RetryQueueVerticle.class);
	private static final Supplier<Boolean> THROTTLE_DISABLED = Suppliers
			.memoize(() -> "true".equals(System.getProperty("throttle.disabled", "false")));

	private final String topic;
	private final RetryProcessor rp;

	public interface RetryProcessor {

		void retry(JsonObject js) throws Exception; // NOSONAR
	}

	private final RetryQueue persistentQueue;
	private final Counter writeCounter;
	private final RateLimiter limitFlushes;

	private MessageConsumer<JsonObject> cons;
	private MessageConsumer<?> compactCons;

	private Future<String> depFuture;

	private static final String TOPIC_TAG = "topic";
	private static final String RETRY_ID = "retry";

	protected RetryQueueVerticle(String topic, RetryProcessor rp) {
		this.topic = topic;
		this.rp = rp;
		this.persistentQueue = new KeydbQueue(topic);
		var reg = MetricsRegistry.get();
		var idFactory = new IdFactory(RETRY_ID, reg, RetryRequester.class);
		this.writeCounter = reg.counter(idFactory.name("writes", TOPIC_TAG, topic));

		this.limitFlushes = RateLimiter.create(1.0);
	}

	public String topic() {
		return topic;
	}

	@Override
	public void start() throws Exception {
		this.depFuture = vertx.deployVerticle(() -> new FlushCompanionVerticle(persistentQueue, rp, topic),
				new DeploymentOptions().setInstances(1).setThreadingModel(ThreadingModel.WORKER));

		EventBus eb = vertx.eventBus();
		final String flushTrigger = "retry.flush." + topic;
		final String flushPayload = "F";
		AtomicLong debounce = new AtomicLong();
		this.cons = eb.consumer("retry." + topic, (Message<JsonObject> msg) -> {
			String jsStr = msg.body().encode();
			persistentQueue.writer().write(jsStr);
			writeCounter.increment();
			logger.debug("WRITE {}", topic);

			vertx.cancelTimer(debounce.get());
			if (THROTTLE_DISABLED.get() || limitFlushes.tryAcquire()) {
				eb.send(flushTrigger, flushPayload);
			} else {
				debounce.set(vertx.setTimer(250, tid -> eb.send(flushTrigger, flushPayload)));
			}
			msg.reply(0L);
		});

		this.compactCons = eb.consumer("retry.compact." + topic, msg -> {
			persistentQueue.compact();
			msg.reply(0L);
		});

		SoftReset.register(() -> {
			vertx.cancelTimer(debounce.get());
		});

	}

	@Override
	public void stop() throws Exception {
		cons.unregister();
		compactCons.unregister();
		if (depFuture.succeeded()) {
			vertx.undeploy(depFuture.result());
		}
		super.stop();
	}

	private static class FlushCompanionVerticle extends AbstractVerticle {
		private static final Logger logger = LoggerFactory.getLogger(FlushCompanionVerticle.class);

		private final RetryQueue persistentQueue;
		private final Timer flushesTimer;
		private final Counter flushedItemsCounter;
		private final String topic;
		private final RetryProcessor rp;
		private MessageConsumer<?> cons;
		private long periodic;
		private final AtomicBoolean flushing = new AtomicBoolean();

		public FlushCompanionVerticle(RetryQueue persistentQueue, RetryProcessor rp, String topic) {
			this.persistentQueue = persistentQueue;
			this.rp = rp;
			var reg = MetricsRegistry.get();
			IdFactory idFactory = new IdFactory(RETRY_ID, reg, RetryRequester.class);
			this.flushesTimer = reg.timer(idFactory.name("flushes", TOPIC_TAG, topic));
			this.flushedItemsCounter = reg.counter(idFactory.name("flushedItems", TOPIC_TAG, topic));
			this.topic = topic;

			SoftReset.register(() -> {
				persistentQueue.clear();
				try {
					stop();
					start();
				} catch (Exception e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
			});
		}

		@Override
		public void start() throws Exception {
			this.cons = vertx.eventBus().consumer("retry.flush." + topic, msg -> {
				try {
					flushRetries();
				} catch (Exception e) {
					logger.error(e.getMessage(), e);
				}
			});

			this.periodic = VertxPlatform.executeBlockingPeriodic(vertx, 60_000, tid -> persistentQueue.compact());
		}

		@Override
		public void stop() throws Exception {
			vertx.cancelTimer(periodic);
			cons.unregister();
			super.stop();
		}

		private void flushRetries() {
			if (!flushing.compareAndSet(false, true)) {
				logger.warn("Flush already in progress");
				return;
			}
			Tailer reader = persistentQueue.reader();
			Stopwatch st = Stopwatch.createStarted();
			CompletableFuture<Long> proc = new CompletableFuture<>();
			Context compCtx = VertxContext.getOrCreateDuplicatedContext();
			untilStopped(proc, compCtx, reader, 0L);
			proc.whenComplete((v, ex) -> {
				flushesTimer.record(st.elapsed(TimeUnit.NANOSECONDS), TimeUnit.NANOSECONDS);
				if (ex != null) {
					flushedItemsCounter.increment(v.longValue());
				}
				flushing.compareAndSet(true, false);
			});
		}

		private void untilStopped(CompletableFuture<Long> proc, Context compCtx, Tailer reader, long l) {
			QueueRecord rec = reader.next();
			if (rec == null) {
				compCtx.runOnContext(v -> proc.complete(l));
			} else {
				CompletableFuture<Void> inVt = new CompletableFuture<>();
				Thread.ofVirtual().name("flush:" + topic).start(() -> {
					try {
						rp.retry(new JsonObject(rec.payload()));
						compCtx.runOnContext(v -> inVt.complete(null));
					} catch (Exception e) {
						compCtx.runOnContext(v -> inVt.completeExceptionally(e));
					}
				});
				inVt.whenComplete((v, ex) -> {
					if (ex != null) {
						proc.complete(l);
					} else {
						reader.commit();
						untilStopped(proc, compCtx, reader, l + 1);
					}
				});
			}
		}

	}

}
