/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2023
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.retry.support;

import java.time.Duration;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.netflix.spectator.api.Clock;
import com.netflix.spectator.api.Id;
import com.netflix.spectator.api.Registry;
import com.netflix.spectator.api.Timer;

import io.vertx.core.eventbus.DeliveryOptions;
import io.vertx.core.eventbus.EventBus;
import io.vertx.core.eventbus.MessageProducer;
import io.vertx.core.json.JsonObject;
import net.bluemind.metrics.registry.IdFactory;
import net.bluemind.metrics.registry.MetricsRegistry;

public class RetryRequester {
	private static final Logger logger = LoggerFactory.getLogger(RetryRequester.class);
	private final Registry reg;
	private final IdFactory idFactory;
	private final Timer timer;
	private final Clock clock;
	private MessageProducer<JsonObject> producer;
	private MessageProducer<JsonObject> compactProducer;

	public RetryRequester(EventBus eventBus, String topic) {
		this.producer = eventBus.publisher("retry." + topic,
				new DeliveryOptions().setSendTimeout(Duration.ofSeconds(10).toMillis()));
		this.compactProducer = eventBus.publisher("retry.compact." + topic,
				new DeliveryOptions().setSendTimeout(Duration.ofSeconds(10).toMillis()));
		this.reg = MetricsRegistry.get();
		this.idFactory = new IdFactory("retry", reg, RetryRequester.class);
		Id timingId = idFactory.name("requester.timing", "topic", topic);
		this.timer = reg.timer(timingId);
		this.clock = reg.clock();
	}

	public void compact() {
		try {
			compactProducer.write(null).toCompletionStage().toCompletableFuture().get(5, TimeUnit.SECONDS);
		} catch (InterruptedException | ExecutionException | TimeoutException e) {
			logger.error("compact timedout", e);
		}
	}

	public void request(JsonObject js) {
		long start = clock.monotonicTime();
		try {
			producer.write(js).andThen(ar -> {
				long delta = clock.monotonicTime() - start;
				timer.record(delta, TimeUnit.NANOSECONDS);
			}).toCompletionStage().toCompletableFuture().get(5, TimeUnit.SECONDS);
		} catch (InterruptedException | ExecutionException | TimeoutException e) {
			logger.error("request timedout", e);
		}

	}

}
