/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

goog.provide('net.bluemind.restclient.SockJsRestClient');
goog.require('goog.Timer');

/**
 * @constructor
 * @param {string} url
 */
net.bluemind.restclient.SockJsRestClient = function (url) {
  this.url_ = url;
  this.sockJSConn_ = null;
  this.handlerMap_ = {};
  this.replyHandlers_ = {};
  this.eventHandlers_ = {};
  this.state = net.bluemind.restclient.SockJsRestClient.CLOSED;
  this.connectionAttempts = 0;
  this.pendingConnection_ = null;
  this.handoverTimerId_ = null;
  this.processedEventIds_ = {};
  this.dedupClearTimerId_ = null;
}

/** @const {number} */
net.bluemind.restclient.SockJsRestClient.HANDOVER_INTERVAL_MS = 60 * 60 * 1000;

/**
 * @type {string}
 * @private
 */
net.bluemind.restclient.SockJsRestClient.prototype.url_ = null;

/**
 * @type {SockJS}
 * @private
 */
net.bluemind.restclient.SockJsRestClient.prototype.sockJSConn_ = null;

/**
 * @type {Array}
 */
net.bluemind.restclient.SockJsRestClient.prototype.handlerMap_ = null;

/**
 * @type {Array}
 */
net.bluemind.restclient.SockJsRestClient.prototype.replyHandlers_ = [];

/**
 * @type {boolean}
 */
net.bluemind.restclient.SockJsRestClient.prototype.connected_ = undefined;

/**
 * @type {number}
 */
net.bluemind.restclient.SockJsRestClient.prototype.connectionAttempts = 0;

/**
 * @export
 */
net.bluemind.restclient.SockJsRestClient.prototype.start = function () {
  this.listeners_ = [];
  this.doConnect_();
}

/**
 * @export
 */
net.bluemind.restclient.SockJsRestClient.prototype.sendMessage = function (restRequest, responseHandler) {
  if (this.sockJSConn_ == null || this.sockJSConn_.readyState !== SockJS.OPEN) {
    if (responseHandler != null) {
      var resp = {
        "body": { "errorCode": "FAILURE", errorType: "ServerFault", message: "restClient not available" },
        "statusCode": 418
      };
      responseHandler(resp);
    } else {
      console.log("try to send msg but sockJS not available", restRequest);
    }
    return;
  }
  if (responseHandler) {
    restRequest['requestId'] = net.bluemind.restclient.SockJsRestClient.makeUUID();
  }

  if (!restRequest['headers']) {
    restRequest['headers'] = {};
  }

  if (!restRequest['headers']["X-BM-ApiKey"]) {
    restRequest['headers']["X-BM-ApiKey"] = goog.global["bmcSessionInfos"]["sid"];
  }

  var str = JSON.stringify(restRequest);

  if (responseHandler) {
    if (restRequest["method"] == "register") {
      this.eventHandlers_[restRequest['path']] = net.bluemind.restclient.SockJsRestClient.copyHandlers(this.eventHandlers_[restRequest['path']], responseHandler);
    } else if (restRequest["method"] == "unregister") {
      delete this.eventHandlers_[restRequest['path']];
    } else {
      this.replyHandlers_[restRequest['requestId']] = responseHandler;
    }
  }
  this.sockJSConn_.send(str);
}

/** 
 * @private
 * param {array} handlers
 */
net.bluemind.restclient.SockJsRestClient.copyHandlers = function (handlers, handler) {
  if (!handlers) {
    handlers = [];
  }
  var copy = handlers.slice(0);
  copy.push(handler);
  return copy;
}
/**
 * @export
 * @param {function()} listener
 */
net.bluemind.restclient.SockJsRestClient.prototype.addListener = function (listener) {
  this.listeners_.push(listener);
  listener.apply(null, [this.connected_]);
}

/**
 * @export
 * @return {boolean}
 */
net.bluemind.restclient.SockJsRestClient.prototype.online = function () {
  return this.connected_;
}

/**
 * @private
 */
net.bluemind.restclient.SockJsRestClient.prototype.doConnect_ = function () {
  if (this.state = net.bluemind.restclient.SockJsRestClient.CONNECTING) {
    return;
  }
  this.connectionAttempts += 1;
  this.state = net.bluemind.restclient.SockJsRestClient.CONNECTING;
  this.sockJSConn_ = this.createConnection_('active');
  console.log("restclient : connecting...");
}

/**
 * @private
 * @param {string} type
 * @return {SockJS}
 */
net.bluemind.restclient.SockJsRestClient.prototype.createConnection_ = function (type) {
  var that = this;
  var sockJSConn = new SockJS(this.url_, undefined, undefined);

  sockJSConn.onopen = function () {
    that.handleOpen_(sockJSConn, type);
  };

  sockJSConn.onclose = function () {
    that.handleClose_(sockJSConn);
  };

  sockJSConn.onmessage = function (e) {
    that.handleMessage_(e);
  };

  return sockJSConn;
}

/**
 * @private
 * @param {SockJS} connection
 * @param {string} type
 */
net.bluemind.restclient.SockJsRestClient.prototype.handleOpen_ = function (connection, type) {
  console.log("restclient: connection open (type: " + type + ")");
  this.state = net.bluemind.restclient.SockJsRestClient.OPEN;

  if (type === 'active') {
    this.handleConnect_();
  } else {
    console.log("Handover: new connection is ready. Swapping.");
    var oldConnection = this.sockJSConn_;

    this.sockJSConn_ = this.pendingConnection_;
    this.pendingConnection_ = null;

    if (oldConnection) {
      oldConnection.close();
    }

    this.setupHeartbeat_();
    this.scheduleHandover_();
  }
};

/**
 * @private
 * @param {MessageEvent} e
 */
net.bluemind.restclient.SockJsRestClient.prototype.handleMessage_ = function (e) {
  var msg = e.data;
  var json = JSON.parse(msg);
  var id = json["requestId"];

  var handler = this.replyHandlers_[id];
  if (handler) {
    delete this.replyHandlers_[id];
    handler(json);
    return;
  }

  var eventHandlers = this.eventHandlers_[json['path']];
  if (eventHandlers) {
    // Deduplication is only active during a handover (when a pendingConnection exists)
    if (this.pendingConnection_) {
      var eventId = json.body.id || json.requestId;
      if (this.processedEventIds_[eventId]) {
        console.log("Duplicate event ignored: ", eventId);
        return;
      }
      this.processedEventIds_[eventId] = true;

      if (this.dedupClearTimerId_) goog.Timer.clear(this.dedupClearTimerId_);
      var that = this;
      this.dedupClearTimerId_ = goog.Timer.callOnce(function () {
        that.processedEventIds_ = {};
      }, 20000, this);
    }

    goog.array.forEach(eventHandlers, function (f) {
      f(json["body"]);
    });
  } else {
    console.log("no handler for ", json);
  }
};

/**
 * @private
 * @param {SockJS} connection
 */
net.bluemind.restclient.SockJsRestClient.prototype.handleClose_ = function (connection) {
  console.log("restclient: a connection closed");

  if (connection === this.pendingConnection_) {
    this.pendingConnection_ = null;
    return;
  }

  if (connection === this.sockJSConn_ && !this.pendingConnection_) {
    this.state = net.bluemind.restclient.SockJsRestClient.CLOSED;
    this.sockJSConn_ = null;
    this.handleDisconnect_();
  }
};

/**
 * @private
 */
net.bluemind.restclient.SockJsRestClient.prototype.handleConnect_ = function () {
  if (this.connected_ != true && this.state == net.bluemind.restclient.SockJsRestClient.OPEN) {
    this.connected_ = true;
    this.connectionAttempts = 0;

    this.scheduleHandover_();
    this.setupHeartbeat_();

    var that = this;

    this.sendMessage({
      "method": "GET",
      "path": "/api/auth/ping",
      "headers": {
        "X-BM-ApiKey": goog.global["bmcSessionInfos"]["sid"]
      },
      "params": {}
    }, function (res) {
      if (res["statusCode"] == 200) {
        that.notify_(true);
      } else {
        that.notify_(true, false);
      }
    });

  }
}

/**
 * @private
 */
net.bluemind.restclient.SockJsRestClient.prototype.handleDisconnect_ = function () {
  if (this.handoverTimerId_) {
    goog.Timer.clear(this.handoverTimerId_);
    this.handoverTimerId_ = null;
  }

  if (this.hearbeatId) {
    goog.Timer.clear(this.hearbeatId);
    this.hearbeatId = null;
  }

  console.log("disconnect, rplyHandlers ", this.replyHandlers_);
  goog.object.forEach(this.replyHandlers_, function (h) {
    console.log("send error to ", h);
    var resp = {
      "body": { "errorCode": "FAILURE", errorType: "ServerFault", message: "restClient disconnected" },
      "statusCode": 418
    };
    h(resp);
  });
  this.replyHandlers_ = [];

  try {
    if (this.sockJSConn_) {
      this.sockJSConn_.close();
    }
  } catch (e) {
    console.log("error during close", e);
  }

  if (this.hearbeatId) {
    goog.Timer.clear(this.hearbeatId);
  }

  this.sockJSConn_ = null;
  if (this.connected_ != false) {
    this.connected_ = false;
    this.notify_(false);
  }

  var maxWaitTime = 30000;
  var baseDelay = 1000;
  var factor = 2;
  var delay = baseDelay * Math.pow(factor, this.connectionAttempts);
  var jitter = Math.random() * 1000;
  delay += jitter;
  delay = Math.min(delay, maxWaitTime);

  goog.Timer.callOnce(function () {
    this.doConnect_();
  }, delay, this);
};

/**
 * @private
 */
net.bluemind.restclient.SockJsRestClient.prototype.scheduleHandover_ = function () {
  if (this.handoverTimerId_) {
    goog.Timer.clear(this.handoverTimerId_);
  }
  var that = this;
  this.handoverTimerId_ = goog.Timer.callOnce(function () {
    console.log("Scheduled handover: initiating...");
    that.pendingConnection_ = that.createConnection_('pending');
  }, net.bluemind.restclient.SockJsRestClient.HANDOVER_INTERVAL_MS, this);
};

/**
 * @private
 */
net.bluemind.restclient.SockJsRestClient.makeUUID = function () {
  return "xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx".replace(/[xy]/g, function (a, b) {
    return b = Math.random() * 16, (a == "y" ? b & 3 | 8 : b | 0).toString(16)
  });
}
/**
 * @private
 */
net.bluemind.restclient.SockJsRestClient.prototype.notify_ = function (state) {
  if (state == true) {
    console.log("fire connected to eventbus");
    goog.array.forEach(this.listeners_, function (l) {
      try {
        l.apply(null, [state]);
      } catch (e) {
        console.log("error during going online", e);
      }
    });
  } else {
    console.log("fire disconnected from eventbus ( state " + this.connected_ + ")");
    goog.array.forEach(this.listeners_, function (l) {
      try {
        l.apply(null, [state]);
      } catch (e) {
        console.log("error during going offline", e);
      }
    });
  }
}

/**
 * @private
 */
net.bluemind.restclient.SockJsRestClient.prototype.setupHeartbeat_ = function () {
  if (this.hearbeatId) {
    goog.Timer.clear(this.hearbeatId);
    this.hearbeatId = null;
  }

  var that = this;
  this.sockJSConn_.onheartbeat = function () {
    if (that.hearbeatId) {
      goog.Timer.clear(that.hearbeatId);
    }
    that.hearbeatId = goog.Timer.callOnce(function () {
      console.log("hearbeat fail, disconnect!");
      this.hearbeatId = null;
      this.sockJSConn_.close();
    }, 1000 * 100, that);
  };

  this.sockJSConn_.onheartbeat();
};
