/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2022
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.resource.service.event;

import java.time.LocalDate;
import java.time.ZonedDateTime;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import net.bluemind.calendar.api.IFreebusyUids;
import net.bluemind.calendar.api.VEvent;
import net.bluemind.calendar.api.VEventSeries;
import net.bluemind.calendar.api.VFreebusy.Slot;
import net.bluemind.calendar.api.VFreebusyQuery;
import net.bluemind.calendar.occurrence.OccurrenceHelper;
import net.bluemind.calendar.service.IInCoreFreebusy;
import net.bluemind.core.api.date.BmDateTime;
import net.bluemind.core.api.date.BmDateTimeWrapper;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.resource.api.ResourceDescriptor;

public abstract class AbstractBookingStrategy implements BookingStrategy {

	protected abstract BookkingDecisionResult check(Slot slot, ZonedDateTime slotStart, ZonedDateTime slotEnd,
			ZonedDateTime evtStart, ZonedDateTime evtEnd);

	@Override
	public BookingDecision isBusy(ItemValue<ResourceDescriptor> resource, VEvent vEvent) {
		String cal = IFreebusyUids.getFreebusyContainerUid(resource.uid);
		IInCoreFreebusy fb = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM)
				.instance(IInCoreFreebusy.class, cal);

		BmDateTime lastEvent = vEvent.dtend;
		if (vEvent.rrule != null) {
			Recurrency recurrency = new Recurrency(vEvent);
			lastEvent = recurrency.getEndDate();
			if (!recurrency.isValid()) {
				return BookingDecision.OUT_OF_RANGE;
			}
		}

		if (new BmDateTimeWrapper(lastEvent).isBefore(BmDateTimeWrapper.create(LocalDate.now()))) {
			return BookingDecision.EXPIRED;
		}

		return checkSlots(fb, vEvent, lastEvent);
	}

	private BookingDecision checkSlots(IInCoreFreebusy fb, VEvent vEvent, BmDateTime lastEvent) {
		VFreebusyQuery query = VFreebusyQuery.create(vEvent.dtstart, lastEvent);
		List<Slot> slots = fb.getTimeOptimized(query, vEvent.dtstart, vEvent.dtend).slots;

		List<VEvent> evts = null;
		if (vEvent.rrule == null) {
			evts = Arrays.asList(vEvent);
		} else {
			VEventSeries series = new VEventSeries();
			series.main = vEvent;
			series.occurrences = Collections.emptyList();
			evts = OccurrenceHelper.list(ItemValue.create("", series), vEvent.dtstart, lastEvent);
		}

		for (VEvent evt : evts) {
			for (Slot slot : slots) {
				ZonedDateTime slotStart = new BmDateTimeWrapper(slot.dtstart).toDateTime();
				ZonedDateTime slotEnd = new BmDateTimeWrapper(slot.dtend).toDateTime();

				ZonedDateTime evtStart = new BmDateTimeWrapper(evt.dtstart).toDateTime();
				ZonedDateTime evtEnd = new BmDateTimeWrapper(evt.dtend).toDateTime();

				BookkingDecisionResult check = check(slot, slotStart, slotEnd, evtStart, evtEnd);
				if (check.mode == Mode.STOP) {
					return check.decision;
				}
			}
		}

		return BookingDecision.FREE;
	}

	record BookkingDecisionResult(Mode mode, BookingDecision decision) {
	}

	enum Mode {
		STOP, CONTINUE
	}

}
