/* BEGIN LICENSE
* Copyright © Blue Mind SAS, 2012-2024
*
* This file is part of BlueMind. BlueMind is a messaging and collaborative
* solution.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of either the GNU Affero General Public License as
* published by the Free Software Foundation (version 3 of the License).
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*
* See LICENSE.txt
* END LICENSE
*/

package net.bluemind.repository.provider;

import java.util.Collections;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.stream.Collectors;

import net.bluemind.core.api.DataSourceType;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ContainerUid;
import net.bluemind.core.container.model.DataLocation;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.BmContext;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.domain.api.Domain;
import net.bluemind.eclipse.common.RunnableExtensionLoader;

public class RepositoryProvider {

	private static final ImplIndex index = new ImplIndex(getFactories());

	private static final class ImplIndex {

		private final Map<DataSourceType, Map<Class<?>, Set<IRepositoryFactory<?>>>> byTypeThenByIFace;

		ImplIndex(List<IRepositoryFactory<?>> factoList) {
			byTypeThenByIFace = new EnumMap<>(DataSourceType.class);
			for (IRepositoryFactory<?> f : factoList) {
				Map<Class<?>, Set<IRepositoryFactory<?>>> byIface = byTypeThenByIFace
						.computeIfAbsent(f.targetRepositoryType(), k -> new HashMap<>());
				byIface.computeIfAbsent(f.factoryClass(), klass -> new HashSet<>()).add(f);
			}
		}

		@SuppressWarnings("unchecked")
		<T, F> IRepositoryFactory<T> factory(Class<T> interfaceClass, DataSourceType flavor, Class<F> factoryClass) {
			Map<Class<?>, Set<IRepositoryFactory<?>>> known = index.byTypeThenByIFace.getOrDefault(flavor,
					Collections.emptyMap());
			Set<IRepositoryFactory<?>> setOfIfaces = known.getOrDefault(interfaceClass, Collections.emptySet());
			return (IRepositoryFactory<T>) setOfIfaces.stream().filter(f -> factoryClass.isAssignableFrom(f.getClass()))
					.findFirst().orElseThrow(() -> {
						String avail = setOfIfaces.stream().map(IRepositoryFactory::name)
								.collect(Collectors.joining(","));
						return new NoSuchElementException(
								"No " + flavor + " factory for " + interfaceClass + ". Available: " + avail);
					});
		}

	}

	private RepositoryProvider() {
	}

	private static List<IRepositoryFactory<?>> getFactories() {
		RunnableExtensionLoader<IRepositoryFactory<?>> rel = new RunnableExtensionLoader<>();
		return rel.loadExtensions("net.bluemind.repository.provider", "factory", "factory", "impl");
	}

	public static <T> T instance(Class<T> interfaceClass, BmContext context, DataLocation location) throws ServerFault {
		if (getFactory(interfaceClass, context,
				IDataLocationBoundFactory.class) instanceof IDataLocationBoundFactory<T> locationBound) {
			return locationBound.instance(context, location);
		} else {
			throw new ServerFault(interfaceClass
					+ " is not an IDataLocationBoundFactory and should not be called with a datalocation");
		}
	}

	public static <T> T instance(Class<T> interfaceClass, BmContext context, ContainerUid cUid) throws ServerFault {
		if (getFactory(interfaceClass, context,
				IContainerUidBoundFactory.class) instanceof IContainerUidBoundFactory<T> uidBound) {
			return uidBound.instance(context, cUid);
		} else {
			throw new ServerFault(interfaceClass
					+ " is not an IContainerUidBoundFactory and should not be called with a container uid");
		}
	}

	public static <T> T instance(Class<T> interfaceClass, BmContext context, ItemValue<Domain> domain)
			throws ServerFault {
		if (getFactory(interfaceClass, context, IDomainBoundFactory.class) instanceof IDomainBoundFactory<T> uidBound) {
			return uidBound.instance(context, domain);
		} else {
			throw new ServerFault(
					interfaceClass + " is not an IDomainBoundFactory and should not be called with a domain item");
		}
	}

	public static <T> T instance(Class<T> interfaceClass, BmContext context, DirEntry owner) throws ServerFault {
		if (getFactory(interfaceClass, context, IOwnerBoundFactory.class) instanceof IOwnerBoundFactory<T> uidBound) {
			return uidBound.instance(context, owner);
		} else {
			throw new ServerFault(
					interfaceClass + " is not an IOwnerBoundFactory and should not be called with a dir entry");
		}
	}

	public static <T> T instance(Class<T> interfaceClass, BmContext context, Container container) throws ServerFault {
		if (getFactory(interfaceClass, context,
				IContainerBoundFactory.class) instanceof IContainerBoundFactory<T> containerBoundFactory) {
			return containerBoundFactory.instance(context, container);
		} else {
			throw new ServerFault(
					interfaceClass + " is not an IContainerBoundFactory and should not be called with a container");
		}
	}

	public static <T> T instance(Class<T> interfaceClass, BmContext context, Container first, Container second)
			throws ServerFault {
		if (getFactory(interfaceClass, context,
				IBiContainerBoundFactory.class) instanceof IBiContainerBoundFactory<T> biContainerBoundFactory) {
			return biContainerBoundFactory.instance(context, first, second);
		} else {
			throw new ServerFault(
					interfaceClass + " is not an IContainerBoundFactory and should not be called with a container");
		}
	}

	public static <T> T instance(Class<T> interfaceClass, BmContext context) throws ServerFault {
		if (getFactory(interfaceClass, context,
				IStandaloneFactory.class) instanceof IStandaloneFactory<T> standaloneFactory) {
			return standaloneFactory.instance(context);
		} else {
			throw new ServerFault(
					interfaceClass + " is not an IStandaloneFactory and should not be called as standalone");
		}
	}

	private static <T, F> IRepositoryFactory<T> getFactory(Class<T> interfaceClass, BmContext context,
			Class<F> factoryClass) {
		return index.factory(interfaceClass, context.getStorageFlavor(), factoryClass);
	}
}
