/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.pool.postgresql.unix;

import java.io.IOException;
import java.net.InetAddress;
import java.net.Socket;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;

import javax.net.SocketFactory;

public class UnixSocketFactory extends SocketFactory {
	private static final String SOCKET_FILE_PREFIX = ".s.PGSQL.";
	private static final int DEFAULT_PORT = 5432;

	public UnixSocketFactory() {
	}

	private Path buildSocketFilePath(String socketPath, int port) {
		return Paths.get(socketPath, SOCKET_FILE_PREFIX + port);
	}

	@Override
	public Socket createSocket(String socketPath, int port) throws IOException {
		if (port <= 0) {
			port = DEFAULT_PORT;
		}
		Path socketFile = buildSocketFilePath(socketPath, port);
		if (!Files.exists(socketFile)) {
			throw new IOException("Unix domain socket does not exists: " + socketFile);
		}

		if (!Files.isReadable(socketFile) || !Files.isWritable(socketFile)) {
			throw new IOException("Unix domain socket " + socketFile + " is not writable");
		}
		return new UnixDomainSocket(socketFile);
	}

	@Override
	public Socket createSocket(String socketPath, int port, InetAddress localHost, int localPort) throws IOException {
		throw new UnsupportedOperationException("Can't create a unix socket with InetAddress");
	}

	@Override
	public Socket createSocket(InetAddress host, int port) throws IOException {
		throw new UnsupportedOperationException("Can't create a unix socket with InetAddress");
	}

	@Override
	public Socket createSocket(InetAddress address, int port, InetAddress localAddress, int localPort)
			throws IOException {
		throw new UnsupportedOperationException("Can't create a unix socket with InetAddress");
	}

	@Override
	public Socket createSocket() throws IOException {
		return createSocket("/var/run/postgresql", 5432);
	}
}