/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.pool.postgresql.unix;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;
import java.net.SocketAddress;
import java.net.UnixDomainSocketAddress;
import java.nio.channels.Channels;
import java.nio.channels.SocketChannel;
import java.nio.file.Path;

public class UnixDomainSocket extends Socket {

	private final UnixDomainSocketAddress address;
	private SocketChannel channel;

	public UnixDomainSocket(Path socketPath) throws IOException {
		super();
		this.address = UnixDomainSocketAddress.of(socketPath);
	}

	@Override
	public void connect(SocketAddress endpoint, int timeout) throws IOException {
		if (this.channel != null && this.channel.isConnected()) {
			return;
		}
		this.channel = SocketChannel.open(this.address);
	}

	@Override
	public InputStream getInputStream() throws IOException {
		if (!isConnected()) {
			throw new IOException("Socket is not connected");
		}
		return Channels.newInputStream(this.channel);
	}

	@Override
	public OutputStream getOutputStream() throws IOException {
		if (!isConnected()) {
			throw new IOException("Socket is not connected");
		}
		return Channels.newOutputStream(this.channel);
	}

	@Override
	public boolean isConnected() {
		return this.channel != null && this.channel.isConnected();
	}

	@Override
	public boolean isClosed() {
		return this.channel == null || !this.channel.isOpen();
	}

	@Override
	public void close() throws IOException {
		if (this.channel != null) {
			this.channel.close();
		}
	}

	@Override
	public int getPort() {
		return 0;
	}
}
