/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2024
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.hornetq.client;

import java.time.Duration;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.lettuce.core.api.StatefulRedisConnection;
import io.lettuce.core.api.sync.RedisCommands;
import net.bluemind.hornetq.client.MQ.SharedMap;
import net.bluemind.hornetq.client.MQKeyDB.Codec;
import net.bluemind.hornetq.client.impl.RedisConnection;
import net.bluemind.lifecycle.helper.SoftReset;

public class KeyDBSharedMap<K, V> implements SharedMap<K, V> {
	private static final Logger logger = LoggerFactory.getLogger(KeyDBSharedMap.class);

	private static final String FLUSH_SHARED_MAPS = "flushSharedMaps";
	private static final ConcurrentHashMap<String, AtomicReference<Map<String, String>>> perKeyCache = new ConcurrentHashMap<>();

	private final String mapName;
	private final RedisCommands<String, String> commands;
	private final Codec<K> keyCodec;
	private final Codec<V> valCodec;
	private final AtomicReference<Map<String, String>> cached;

	public KeyDBSharedMap(String name, Codec<K> key, Codec<V> val, RedisConnection redisConnection) {
		this.mapName = name;
		StatefulRedisConnection<String, String> statefullConnection = redisConnection.keyvalue();
		statefullConnection.setTimeout(Duration.ofSeconds(10));
		this.commands = statefullConnection.sync();
		this.keyCodec = key;
		this.valCodec = val;
		this.cached = perKeyCache.computeIfAbsent(mapName, k -> {
			var holder = new AtomicReference<>(commands.hgetall(k));
			redisConnection.registerInvalidationListener(k, () -> {
				logger.info("Refresh {} from server", k);
				holder.set(commands.hgetall(k));
			});
			redisConnection.registerReconnectListener(k, () -> {
				// Needed to re-enable listeners
				commands.get(FLUSH_SHARED_MAPS);
				commands.hgetall(k);
			});

			return holder;
		});

		// Get key to get future invalidation events
		// Useful for JUnit see net.bluemind.tests.defaultdata.KeydbHelper
		commands.get(FLUSH_SHARED_MAPS);
		redisConnection.registerInvalidationListener(FLUSH_SHARED_MAPS, () -> {
			logger.info("Flush local shared map");
			perKeyCache.clear();
		});

		SoftReset.register(() -> {
			perKeyCache.clear();
			cached.get().clear();
		});
	}

	@Override
	public void put(K key, V value) {
		String encKey = keyCodec.toString(key);
		String encVal = valCodec.toString(value);
		commands.hset(mapName, encKey, encVal);
		cached.get().put(encKey, encVal);
	}

	@Override
	public V get(K key) {
		String val = cached.get().get(keyCodec.toString(key));
		return val == null ? null : valCodec.fromString(val);
	}

	@Override
	public Set<K> keys() {
		return cached.get().keySet().stream().map(keyCodec::fromString).collect(Collectors.toSet());
	}

	@Override
	public void remove(K key) {
		String encKey = keyCodec.toString(key);
		commands.hdel(mapName, encKey);
		cached.get().remove(encKey);
	}
}
