/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2024
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.hornetq.client;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.function.Predicate;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.type.TypeReference;

import io.vertx.core.json.JsonObject;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.core.utils.JsonUtils.ValueReader;
import net.bluemind.hornetq.client.MQKeyDB.Codec;
import net.bluemind.hornetq.client.impl.KeyDBClient;

public final class MQ {

	private static final Logger logger = LoggerFactory.getLogger(MQ.class);
	public static final String MEMBERSHIP_EVENTS_ADDRESS = "hazelcast.membership";
	private static final KeyDBClient keyDBClient = new KeyDBClient();

	@FunctionalInterface
	public interface IMQConnectHandler {

		void connected();

		default void connectionFailed(Throwable t) {
			logger.error("KeyDB connect failed", t);
		}
	}

	public static CompletableFuture<Void> init() {
		return keyDBClient.init();
	}

	public static final synchronized void init(final IMQConnectHandler handler) {
		keyDBClient.init(handler);
	}

	public static interface SharedMap<K, V> {
		void put(K key, V value);

		V get(K key);

		default void putAll(Map<K, V> map) {
			map.forEach(this::put);
		}

		Set<K> keys();

		void remove(K key);

		default Map<K, V> asMap() {
			// Can't use Collectors.toMap as value may be null
			// https://bugs.openjdk.java.net/browse/JDK-8148463
			return keys().stream().collect(HashMap::new, (map, key) -> map.put(key, get(key)), HashMap::putAll);
		}
	}

	private static MQKeyDB getOrFail() {
		try {
			return keyDBClient.keyDBStart.get();
		} catch (InterruptedException ie) {
			Thread.currentThread().interrupt();
			throw new KeyDBNotAvailableException(ie);
		} catch (ExecutionException e) {
			throw new KeyDBNotAvailableException(e);
		}
	}

	@SuppressWarnings("serial")
	private static class KeyDBNotAvailableException extends RuntimeException {
		KeyDBNotAvailableException(Throwable t) {
			super("NO: MQKeyDB is not available: ", t);
		}
	}

	public static SharedMap<String, String> sharedMap(String name) {
		return getOrFail().sharedMap(name);
	}

	public static <C> SharedMap<String, C> sharedMap(String name, Class<C> forClass) {
		return getOrFail().sharedMap(name, forClass);
	}

	public static <C> SharedMap<String, C> sharedMap(String name, TypeReference<C> forClass) {
		ValueReader<C> reader = JsonUtils.reader(forClass);

		return getOrFail().sharedMap(name, Codec.STR, new Codec<C>() {

			@Override
			public C fromString(String s) {
				return reader.read(s);
			}

			@Override
			public String toString(C value) {
				return JsonUtils.asString(value);
			}
		});
	}

	public static long clusterTime() {
		// not yet with keydb
		return System.currentTimeMillis();
	}

	public static Consumer registerConsumer(String topic, OutOfProcessMessageHandler handler) {

		return getOrFail().registerConsumer(topic, null, msg -> {
			try {
				handler.handle(msg);
			} catch (Exception e) {
				logger.error("unhandled exception @{}: {}", topic, e);
			}
		});
	}

	public static Consumer registerConsumer(String topic, Predicate<JsonObject> filter,
			OutOfProcessMessageHandler handler) {

		return getOrFail().registerConsumer(topic, filter, msg -> {
			try {
				handler.handle(msg);
			} catch (Exception e) {
				logger.error("unhandled exception @{}: {}", topic, e);
			}
		});
	}

	public static Producer registerProducer(String topic) {
		return getOrFail().registerProducer(topic);
	}

	public static Producer getProducer(String topic) {
		return getOrFail().getProducer(topic);
	}

	public static OOPMessage newMessage() {
		return new OOPMessage(new JsonObject());
	}

	public static List<String> topics() {
		return getOrFail().getProducersTopics().keySet().stream().distinct().toList();
	}
}
