/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2023
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.webmodule.authenticationfilter;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.Vertx;
import net.bluemind.authentication.api.IAuthenticationPromise;
import net.bluemind.authentication.api.LoginResponse;
import net.bluemind.authentication.api.LoginResponse.Status;
import net.bluemind.config.Token;
import net.bluemind.core.api.AsyncHandler;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.http.HttpClientProvider;
import net.bluemind.core.rest.http.ILocator;
import net.bluemind.core.rest.http.VertxPromiseServiceProvider;
import net.bluemind.domain.api.IDomainsPromise;
import net.bluemind.keydb.sessiondata.SessionData;
import net.bluemind.keydb.sessiondata.SessionDataStore;
import net.bluemind.mailbox.api.IMailboxesPromise;
import net.bluemind.mailbox.api.Mailbox;
import net.bluemind.mailbox.api.Mailbox.Type;
import net.bluemind.network.topology.Topology;
import net.bluemind.server.api.TagDescriptor;
import net.bluemind.webmodule.authenticationfilter.internal.Credentials;
import net.bluemind.webmodule.authenticationfilter.internal.Credentials.CredentialBuilderValidationException;

public class AuthProvider {
	private static final Logger logger = LoggerFactory.getLogger(AuthProvider.class);

	public static final int DEFAULT_MAX_SESSIONS_PER_USER = 5;
	private static final String BM_WEBSERVER_AUTHFILTER = "bm-webserver-authfilter";

	private HttpClientProvider clientProvider;

	private final String domainUid;
	private final boolean internalAuth;

	public AuthProvider(Vertx vertx, String domainUid) {
		clientProvider = new HttpClientProvider(vertx);
		this.domainUid = domainUid;
		this.internalAuth = false;
	}

	public AuthProvider(Vertx vertx, String domainUd, boolean internalAuth) {
		clientProvider = new HttpClientProvider(vertx);
		this.domainUid = domainUd;
		this.internalAuth = internalAuth;
	}

	public void sessionId(Credentials credentials, List<String> remoteIps, AsyncHandler<SessionData> handler) {
		if ("admin0@global.virt".equals(credentials.getLoginAtDomain())) {
			if (internalAuth) {
				doSudo(remoteIps, credentials, handler);
			} else {
				// No admin0@global.virt authentication using external authentication service
				handler.failure(new ServerFault(
						"Authentication failure: admin0@global.virt only available from internal authentication",
						ErrorCode.FORBIDDEN));
			}

			return;
		}

		externalCredsToMailbox(credentials, remoteIps, handler);
	}

	public void sessionId(final String loginAtDomain, final String password, List<String> remoteIps,
			final AsyncHandler<SessionData> handler) {
		VertxPromiseServiceProvider sp = getProvider(null, remoteIps);

		logger.info("authenticating {}", loginAtDomain);
		IAuthenticationPromise auth = sp.instance(TagDescriptor.bm_core.getTag(), IAuthenticationPromise.class);
		auth.loginWithParams(loginAtDomain.toLowerCase(), password, BM_WEBSERVER_AUTHFILTER, true).exceptionally(e -> {
			logger.error("error during authentication of {}", loginAtDomain, e);
			handler.failure(new ServerFault("error login: No server assigned or server not avalaible"));
			return null;
		}).thenAccept(lr -> {
			logger.info("Authenticated {}, response: {}", loginAtDomain, lr.status);
			if (lr.status == Status.Ok || lr.status == Status.Expired) {
				handlerLoginSuccess(lr, handler);
			} else {
				handler.failure(new ServerFault("error during login " + lr.message, ErrorCode.INVALID_PASSWORD));
			}
		});
	}

	private void doSudo(List<String> remoteIps, Credentials credentials, final AsyncHandler<SessionData> handler) {
		getProvider(Token.admin0(), remoteIps).instance(IAuthenticationPromise.class)
				.suWithParams(credentials.getLoginAtDomain(), true).exceptionally(t -> null).thenAccept(lr -> {
					if (lr == null) {
						handler.failure(
								new ServerFault("Error during sudo for user " + credentials.getLoginAtDomain()));
						return;
					}

					if (lr.status == Status.Ok) {
						handlerLoginSuccess(lr, handler);
					} else {
						handler.success(null);
					}
				});
	}

	private void externalCredsToMailbox(Credentials credentials, List<String> remoteIps,
			AsyncHandler<SessionData> handler) {
		VertxPromiseServiceProvider provider = getProvider(Token.admin0(), remoteIps);
		CompletableFuture<ItemValue<Mailbox>> mailboxFuture;
		if (credentials.getUserUid() != null) {
			logger.info("Getting mailbox uid: {}", credentials.getUserUid());
			mailboxFuture = provider.instance(IMailboxesPromise.class, credentials.getDomainUid())
					.getComplete(credentials.getUserUid());
		} else if (credentials.getLoginAtDomain() != null) {
			logger.info("Searching mailbox with email: {}", credentials.getLoginAtDomain());
			// BM-21397: don't search byName, only byEmail
			mailboxFuture = provider.instance(IMailboxesPromise.class, credentials.getDomainUid())
					.byEmail(credentials.getLoginAtDomain());
		} else {
			handler.failure(
					new ServerFault("Unable to get User: undefined userUid and loginAtDomin", ErrorCode.FORBIDDEN));
			return;
		}

		mailboxFuture.whenComplete((mailbox, mailboxException) -> {
			if (mailboxException != null) {
				handler.failure(mailboxException);
				return;
			}

			if (mailbox != null) {
				sudoFromMailbox(credentials, mailbox, remoteIps, handler);
				return;
			}

			// on-the-fly import need defined loginAtDomain
			if (credentials.getLoginAtDomain() != null) {
				tryOnTheFlyimport(credentials, provider, remoteIps, handler);
				return;
			}

			handler.failure(
					new ServerFault("Unable to try on the fly import: undefined loginAtDomain", ErrorCode.FORBIDDEN));
		});
	}

	private void tryOnTheFlyimport(Credentials credentials, VertxPromiseServiceProvider provider,
			List<String> remoteIps, AsyncHandler<SessionData> handler) {
		if (internalAuth || credentials.getDomainUid().equals(credentials.getLoginAtDomain())) {
			// Internal auth or domain login part equals credential domain UID, guess
			// credential
			logger.info("Try sudo with login {} (try on-the-fly import)", credentials.getLoginAtDomain());
			doSudo(remoteIps, credentials, handler);
			return;
		}

		// Check that login domain part is same domain as credential domain
		// UID
		provider.instance(IDomainsPromise.class).findByNameOrAliases(credentials.getLoginDomainPart())
				.whenComplete((domain, domainException) -> {
					if (domainException != null) {
						handler.failure(domainException);
						return;
					}

					// BM-21416: Ensure external credentials domain match BlueMind domain before
					// trying on-the-fly import
					if (!domain.uid.equals(credentials.getDomainUid())) {
						handler.failure(new ServerFault("Authentication failure: external credentials "
								+ credentials.getLoginAtDomain() + " not from domain: " + domainUid,
								ErrorCode.FORBIDDEN));
						return;
					}

					logger.info("Try sudo with login {} (try on-the-fly import)", credentials.getLoginAtDomain());
					doSudo(remoteIps, credentials, handler);
				});
	}

	private void sudoFromMailbox(Credentials externalCreds, ItemValue<Mailbox> mailbox, List<String> remoteIps,
			AsyncHandler<SessionData> handler) {
		if (mailbox.value.type != Type.user || mailbox.value.archived) {
			handler.success(null);
			return;
		}

		String realLoginAtdomain = mailbox.value.name + "@" + externalCreds.getDomainUid();

		logger.info("Try sudo with login {} ({})", realLoginAtdomain, externalCreds);
		try {
			doSudo(remoteIps,
					Credentials.Builder.fromExternalCred(externalCreds).setLoginAtDomain(realLoginAtdomain).build(),
					handler);
		} catch (CredentialBuilderValidationException e) {
			handler.failure(e);
		}
	}

	private void handlerLoginSuccess(LoginResponse lr, AsyncHandler<SessionData> handler) {
		try {
			SessionData sessionData = SessionDataStore.get().getIfPresent(lr.authKey);
			if (sessionData == null) {
				logout(lr.latd, lr.authKey);
				handler.failure(
						new ServerFault("No session data found for SID: " + lr.authKey, ErrorCode.AUTHENTICATION_FAIL));
				return;
			}

			handler.success(sessionData);
		} catch (Exception e) {
			handler.failure(e);
		}
	}

	public CompletableFuture<Void> logout(String sid) {
		SessionData sessionData = SessionDataStore.get().getIfPresent(sid);

		if (sessionData == null) {
			return logout("Unknown", sid);
		}

		if (sessionData.jwtToken == null) {
			// Interactive sessions open from API
			// Not an OpenId Session, invalidate from webserver sessions local cache
			// to avoid requests loop that may occur until the cache expires
			SessionDataStore.get().invalidate(sid);
		}

		return logout(sessionData.loginAtDomain, sessionData.authKey);
	}

	/**
	 * Logout on core as core HZ message will invalidate webserver session
	 * 
	 * See {@link AuthenticationFilter#setVertx(Vertx)}
	 * 
	 * @param id
	 * @param sessionId
	 * @return
	 */
	private CompletableFuture<Void> logout(String id, String sessionId) {
		if (logger.isDebugEnabled()) {
			logger.debug("Log out session {} for {}", sessionId, id);
		}

		return getProvider(sessionId, Collections.emptyList()).instance(IAuthenticationPromise.class).logout()
				.whenComplete((v, fn) -> {
					if (fn != null) {
						logger.warn(fn.getMessage());
					}
				});
	}

	private VertxPromiseServiceProvider getProvider(String apiKey, List<String> remoteIps) {
		ILocator lc = (String service, AsyncHandler<String[]> asyncHandler) -> asyncHandler.success(
				new String[] { Topology.get().anyIfPresent(service).map(s -> s.value.address()).orElse("127.0.0.1") });
		VertxPromiseServiceProvider prov = new VertxPromiseServiceProvider(clientProvider, lc, apiKey, remoteIps);
		prov.setOrigin(BM_WEBSERVER_AUTHFILTER);
		return prov;
	}
}
