/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2017
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.backend.cyrus.partitions;

import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.backend.mail.replica.api.MailboxReplicaRootDescriptor;
import net.bluemind.backend.mail.replica.api.MailboxReplicaRootDescriptor.Namespace;
import net.bluemind.core.api.fault.ServerFault;

public class CyrusBoxes {

	private static final Logger logger = LoggerFactory.getLogger(CyrusBoxes.class);

	private static final Pattern userMboxRootRe = Pattern.compile("(.*)!user\\.([^\\.]*)$");
	private static final Pattern userMboxRe = Pattern.compile("(.*)!user\\.([^\\.]*)\\.(.*)$");
	private static final Pattern deletedMbox = Pattern.compile("(.*)!DELETED.user\\.([^\\.]*)(.*)$");
	private static final Pattern deletedSharedMbox = Pattern.compile("(.*)!DELETED\\.([^\\.]*)\\.(.*)$");
	private static final Pattern sharedMbox = Pattern.compile("(.*)!(.*)$");

	public static class ReplicatedBox {
		public Namespace ns;
		public String local;
		public String partition;
		public String folderName;
		public boolean mailboxRoot;

		public String toString() {
			return "[RB " + ns + " local: " + local + ", part: " + partition + ", f: " + folderName + "]";
		}

		public MailboxReplicaRootDescriptor asDescriptor() {
			MailboxReplicaRootDescriptor rd = new MailboxReplicaRootDescriptor();
			rd.ns = ns;
			rd.name = local;
			return rd;
		}

		public String fullName() {
			switch (ns) {
			case users:
				return folderName;
			case shared:
				return mailboxRoot ? folderName : local.replace('^', '.') + "/" + folderName;
			case deleted:
			case deletedShared:
			default:
				throw new ServerFault("Can't deal with deleted namespace on " + this);
			}
		}
	}

	private CyrusBoxes() {
	}

	/**
	 * Computes partition from domain part & mailbox name from local part of a fully
	 * qualified userName.
	 *
	 * Returns null otherwise.
	 *
	 * @param userName fully qualified user name
	 * @return a {@link ReplicatedBox}
	 */
	public static ReplicatedBox forLoginAtDomain(String userName) {
		ReplicatedBox box = new ReplicatedBox();
		box.ns = Namespace.users;
		int atIdx = userName.indexOf('@');
		if (atIdx < 0) {
			logger.warn("Username without domain part, can't continue.");
			return null;
		}
		box.local = userName.substring(0, atIdx);
		box.partition = userName.substring(atIdx + 1).replace('.', '_');
		box.folderName = "";
		return box;
	}

}
